<?php
/**
 * In-Bot Admin Panel - Complete Management System
 * Updated: 2025-10-15 19:13:08 UTC
 * Created by: saportinsta65
 */

class AdminCommands {
    private $telegram;
    private $db;
    
    public function __construct($telegram, $db) {
        $this->telegram = $telegram;
        $this->db = $db;
    }
    
    /**
     * 🔐 بررسی دسترسی ادمین
     */
    public function isAdmin($user_id) {
        return $user_id == ADMIN_CHAT_ID;
    }
    
    /**
     * 👨‍💼 نمایش پنل ادمین اصلی
     */
    public function showMainPanel($chat_id, $user_id) {
        if (!$this->isAdmin($user_id)) {
            $this->telegram->sendMessage($chat_id, "❌ دسترسی غیرمجاز");
            return;
        }
        
        $stats = $this->db->getAdminStats();
        
        $message = "👨‍💼 <b>پنل مدیریت کاپیتان ترید</b>

━━━━━━━━━━━━━━━━━━━━

📊 <b>آمار لحظه‌ای:</b>

👥 کل کاربران: " . number_format($stats['total_users']) . "
✅ تایید شده: " . number_format($stats['verified_users']) . "
💰 کل درآمد: " . number_format($stats['total_revenue']) . " تومان
⏳ پرداخت انتظار: " . number_format($stats['pending_payments']) . "
📈 ثبت‌نام امروز: " . number_format($stats['today_registrations']) . "

━━━━━━━━━━━━━━━━━━━━

🎯 <b>گزینه مورد نظر را انتخاب کنید:</b>";
        
        $keyboard = $this->getMainAdminKeyboard();
        $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 💳 نمایش پرداخت‌های در انتظار
     */
    public function showPendingPayments($chat_id, $user_id) {
        if (!$this->isAdmin($user_id)) return;
        
        $payments = $this->db->getPendingPayments(10);
        
        if (empty($payments)) {
            $message = "✅ <b>پرداخت در انتظار نداریم!</b>

🎉 همه پرداخت‌ها بررسی شده‌اند.";
            $this->telegram->sendMessage($chat_id, $message, $this->getAdminBackKeyboard());
            return;
        }
        
        $message = "💳 <b>پرداخت‌های در انتظار (" . count($payments) . ")</b>

━━━━━━━━━━━━━━━━━━━━\n\n";
        
        foreach ($payments as $i => $p) {
            $message .= "💳 <b>پرداخت " . ($i + 1) . ":</b>

🆔 کد: <code>{$p['id']}</code>
👤 {$p['first_name']} (@{$p['username']})
💰 " . number_format($p['amount']) . " تومان
📅 " . date('m/d H:i', strtotime($p['created_at'])) . "
🔄 {$p['status']}

━━━━━━━━━━━━━━━━━━━━\n\n";
        }
        
        $message .= "🎯 <b>دستورات:</b>

✅ <code>/ok {ID}</code> - تایید
❌ <code>/no {ID}</code> - رد
🔍 <code>/show {ID}</code> - جزئیات

📋 <b>مثال:</b> <code>/ok {$payments[0]['id']}</code>";
        
        $this->telegram->sendMessage($chat_id, $message, $this->getAdminBackKeyboard());
    }
    
    /**
     * ✅ تایید پرداخت
     */
    public function approvePayment($chat_id, $user_id, $payment_id) {
        if (!$this->isAdmin($user_id)) return;
        
        $this->telegram->sendMessage($chat_id, "🔄 در حال پردازش...");
        
        try {
            $payment = $this->db->getPaymentInfo($payment_id);
            if (!$payment) {
                $this->telegram->sendMessage($chat_id, "❌ پرداخت یافت نشد!");
                return;
            }
            
            // Update payment status
            $this->db->updatePaymentStatus($payment_id, 'completed', null, 'admin_' . $user_id);
            
            // Create license if needed
            $license_result = $this->createLicenseForUser($payment['user_id'], $payment_id);
            
            // Notify user
            $this->notifyUserPaymentApproved($payment, $license_result);
            
            $message = "✅ <b>پرداخت تایید شد!</b>

🆔 کد: <code>{$payment_id}</code>
👤 {$payment['first_name']}
💰 " . number_format($payment['amount']) . " تومان
🎫 لایسنس: " . ($license_result['success'] ? "✅ ایجاد شد" : "❌ خطا") . "

🎯 کاربر اطلاع‌رسانی شد.";
            
            $this->telegram->sendMessage($chat_id, $message, $this->getAdminBackKeyboard());
            
        } catch (Exception $e) {
            error_log("Approve payment error: " . $e->getMessage());
            $this->telegram->sendMessage($chat_id, "❌ خطا در تایید پرداخت!");
        }
    }
    
    /**
     * ❌ رد پرداخت
     */
    public function rejectPayment($chat_id, $user_id, $payment_id, $reason = '') {
        if (!$this->isAdmin($user_id)) return;
        
        try {
            $payment = $this->db->getPaymentInfo($payment_id);
            if (!$payment) {
                $this->telegram->sendMessage($chat_id, "❌ پرداخت یافت نشد!");
                return;
            }
            
            // Update payment status
            $this->db->updatePaymentStatus($payment_id, 'rejected', null, 'admin_' . $user_id);
            
            // Notify user
            $this->notifyUserPaymentRejected($payment, $reason);
            
            $message = "❌ <b>پرداخت رد شد!</b>

🆔 کد: <code>{$payment_id}</code>
👤 {$payment['first_name']}
💰 " . number_format($payment['amount']) . " تومان
📝 دلیل: " . ($reason ?: "مشخص نشده") . "

🎯 کاربر اطلاع‌رسانی شد.";
            
            $this->telegram->sendMessage($chat_id, $message, $this->getAdminBackKeyboard());
            
        } catch (Exception $e) {
            error_log("Reject payment error: " . $e->getMessage());
            $this->telegram->sendMessage($chat_id, "❌ خطا در رد پرداخت!");
        }
    }
    
    /**
     * 🔍 نمایش جزئیات پرداخت
     */
    public function showPaymentDetails($chat_id, $user_id, $payment_id) {
        if (!$this->isAdmin($user_id)) return;
        
        $payment = $this->db->getPaymentInfo($payment_id);
        if (!$payment) {
            $this->telegram->sendMessage($chat_id, "❌ پرداخت یافت نشد!");
            return;
        }
        
        $message = "🔍 <b>جزئیات پرداخت</b>

━━━━━━━━━━━━━━━━━━━━

🆔 <b>کد پرداخت:</b> <code>{$payment['id']}</code>

👤 <b>کاربر:</b>
• نام: {$payment['first_name']}
• یوزرنیم: @{$payment['username']}
• موبایل: {$payment['mobile_number']}
• آیدی: <code>{$payment['user_id']}</code>

💰 <b>پرداخت:</b>
• مبلغ: " . number_format($payment['amount']) . " تومان
• روش: {$payment['payment_method']}
• وضعیت: {$payment['status']}

📅 <b>تاریخ:</b>
• ایجاد: " . date('Y/m/d H:i', strtotime($payment['created_at'])) . "
• آپدیت: " . date('Y/m/d H:i', strtotime($payment['updated_at'])) . "

━━━━━━━━━━━━━━━━━━━━

🎯 <b>اقدامات:</b>
✅ /ok {$payment['id']} - تایید
❌ /no {$payment['id']} - رد";
        
        $this->telegram->sendMessage($chat_id, $message, $this->getAdminBackKeyboard());
        
        // Send receipt if exists
        if ($payment['receipt_image']) {
            $this->telegram->sendPhoto($chat_id, $payment['receipt_image'], "📸 رسید پرداخت {$payment['id']}");
        }
    }
    
    /**
     * 👥 آمار کاربران
     */
    public function showUserStats($chat_id, $user_id) {
        if (!$this->isAdmin($user_id)) return;
        
        $stats = $this->db->getAdminStats();
        $activities = $this->db->getRecentActivities(5);
        
        $message = "👥 <b>آمار کاربران</b>

━━━━━━━━━━━━━━━━━━━━

📊 <b>آمار کلی:</b>
• کل کاربران: " . number_format($stats['total_users']) . "
• تایید شده: " . number_format($stats['verified_users']) . "
• مسدود شده: 0
• فعال امروز: " . number_format($stats['today_registrations']) . "
• رشد هفتگی: " . number_format($stats['weekly_growth']) . "

💰 <b>آمار مالی:</b>
• کل درآمد: " . number_format($stats['total_revenue']) . " تومان
• پرداخت موفق: " . number_format($stats['total_payments']) . "
• در انتظار: " . number_format($stats['pending_payments']) . "

━━━━━━━━━━━━━━━━━━━━

📋 <b>آخرین فعالیت‌ها:</b>\n";
        
        foreach ($activities as $activity) {
            $time = date('H:i', strtotime($activity['created_at']));
            $message .= "• $time - {$activity['first_name']}: " . substr($activity['action'], 0, 30) . "...\n";
        }
        
        $this->telegram->sendMessage($chat_id, $message, $this->getAdminBackKeyboard());
    }
    
    /**
     * 📢 ارسال پیام همگانی
     */
    public function startBroadcast($chat_id, $user_id) {
        if (!$this->isAdmin($user_id)) return;
        
        $this->db->setUserState($user_id, 'admin_broadcast');
        
        $message = "📢 <b>ارسال پیام همگانی</b>

━━━━━━━━━━━━━━━━━━━━

📝 <b>متن پیام خود را بنویسید:</b>

💡 <b>راهنمایی:</b>
• از HTML و emoji استفاده کنید
• پیام کوتاه و مفید باشد
• بعد از ارسال تایید می‌خواهم

✍️ <b>پیام را در پیام بعدی بفرستید:</b>";
        
        $this->telegram->sendMessage($chat_id, $message, $this->getAdminBackKeyboard());
    }
    
    /**
     * 📤 پردازش پیام همگانی
     */
    public function processBroadcast($chat_id, $user_id, $message_text) {
        if (!$this->isAdmin($user_id)) return;
        
        $user_count = $this->db->getAdminStats()['total_users'];
        
        $confirm_message = "📢 <b>تایید ارسال پیام همگانی</b>

━━━━━━━━━━━━━━━━━━━━

📝 <b>پیام شما:</b>
{$message_text}

━━━━━━━━━━━━━━━━━━━━

👥 <b>دریافت‌کنندگان:</b> " . number_format($user_count) . " کاربر

⚠️ <b>آیا مطمئن هستید؟</b>";
        
        $keyboard = [
            'keyboard' => [
                [
                    ['text' => '✅ ارسال کن'],
                    ['text' => '❌ انصراف']
                ],
                [
                    ['text' => '👨‍💼 پنل مدیریت']
                ]
            ],
            'resize_keyboard' => true,
            'one_time_keyboard' => false
        ];
        
        // Save broadcast message temporarily
        $this->db->setUserState($user_id, 'admin_broadcast_confirm');
        // TODO: Save message in temporary storage
        
        $this->telegram->sendMessage($chat_id, $confirm_message, $keyboard);
    }
    
    /**
     * 🎹 کیبورد اصلی ادمین
     */
    private function getMainAdminKeyboard() {
        return [
            'keyboard' => [
                [
                    ['text' => '💳 بررسی پرداخت‌ها'],
                    ['text' => '👥 آمار کاربران']
                ],
                [
                    ['text' => '📢 پیام همگانی'],
                    ['text' => '⚙️ تنظیمات']
                ],
                [
                    ['text' => '🔙 منوی اصلی']
                ]
            ],
            'resize_keyboard' => true,
            'one_time_keyboard' => false
        ];
    }
    
    /**
     * 🔙 کیبورد بازگشت ادمین
     */
    private function getAdminBackKeyboard() {
        return [
            'keyboard' => [
                [
                    ['text' => '👨‍💼 پنل مدیریت'],
                    ['text' => '🔙 منوی اصلی']
                ]
            ],
            'resize_keyboard' => true,
            'one_time_keyboard' => false
        ];
    }
    
    /**
     * 🎫 ایجاد لایسنس برای کاربر
     */
    private function createLicenseForUser($user_id, $payment_id) {
        try {
            // TODO: Integrate with SpotPlayer API
            $license_data = [
                'license_id' => 'lic_' . time() . '_' . $user_id,
                'license_key' => 'DEMO_' . strtoupper(bin2hex(random_bytes(20))),
                'download_url' => 'https://spotplayer.ir/download',
                'course_ids' => [COURSE_ID],
                'is_test' => false
            ];
            
            $this->db->createSpotPlayerLicense($user_id, $payment_id, $license_data);
            
            return ['success' => true, 'license_data' => $license_data];
            
        } catch (Exception $e) {
            error_log("Create license error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * 📧 اطلاع‌رسانی تایید پرداخت به کاربر
     */
    private function notifyUserPaymentApproved($payment, $license_result) {
        $message = "🎉 <b>تبریک! پرداخت شما تایید شد!</b>

━━━━━━━━━━━━━━━━━━━━

✅ <b>پرداخت با موفقیت تکمیل شد</b>

🆔 کد پیگیری: <code>{$payment['id']}</code>
💰 مبلغ: " . number_format($payment['amount']) . " تومان

━━━━━━━━━━━━━━━━━━━━";
        
        if ($license_result['success']) {
            $license = $license_result['license_data'];
            $message .= "

🎫 <b>لایسنس شما آماده است:</b>
<code>{$license['license_key']}</code>

📱 <b>لینک دانلود:</b>
{$license['download_url']}

━━━━━━━━━━━━━━━━━━━━

🎯 <b>مراحل بعدی:</b>
1️⃣ اپلیکیشن SpotPlayer را دانلود کنید
2️⃣ لایسنس را در اپ وارد کنید
3️⃣ از تمام ویدیوها لذت ببرید

💎 <b>به خانواده کاپیتان ترید خوش آمدید!</b>";
        } else {
            $message .= "

⚠️ <b>لایسنس در حال آماده‌سازی است</b>
تا چند ساعت آینده برایتان ارسال می‌شود.

📞 پشتیبانی: " . SUPPORT_USERNAME;
        }
        
        $this->telegram->sendMessage($payment['chat_id'], $message);
    }
    
    /**
     * 📧 اطلاع‌رسانی رد پرداخت به کاربر
     */
    private function notifyUserPaymentRejected($payment, $reason) {
        $message = "❌ <b>پرداخت شما تایید نشد</b>

━━━━━━━━━━━━━━━━━━━━

🆔 کد پیگیری: <code>{$payment['id']}</code>
💰 مبلغ: " . number_format($payment['amount']) . " تومان

📝 <b>دلیل:</b> " . ($reason ?: "رسید نامعتبر یا ناواضح") . "

━━━━━━━━━━━━━━━━━━━━

🔄 <b>راه‌حل:</b>
• رسید واضح‌تری ارسال کنید
• یا با پشتیبانی تماس بگیرید

📞 پشتیبانی: " . SUPPORT_USERNAME . "

💡 <b>می‌توانید دوباره تلاش کنید</b>";
        
        $this->telegram->sendMessage($payment['chat_id'], $message);
    }
}
?>