<?php
/**
 * Enhanced Payment Processing System
 * Captain Trade Bot - Professional Payment Management
 * Updated: 2025-10-15 16:33:51 UTC
 * Created by: saportinsta65
 */

/**
 * ZarinPal Payment Gateway Integration
 */
class ZarinPalPayment {
    private $merchant_id;
    private $sandbox;
    private $api_url;
    private $gateway_url;
    
    public function __construct() {
        $this->merchant_id = ZARINPAL_MERCHANT_ID;
        $this->sandbox = ZARINPAL_SANDBOX;
        
        if ($this->sandbox) {
            $this->api_url = 'https://sandbox.zarinpal.com/pg/rest/WebGate/';
            $this->gateway_url = 'https://sandbox.zarinpal.com/pg/StartPay/';
        } else {
            $this->api_url = 'https://www.zarinpal.com/pg/rest/WebGate/';
            $this->gateway_url = 'https://www.zarinpal.com/pg/StartPay/';
        }
    }
    
    /**
     * Create payment request with enhanced features
     */
    public function createPayment($user_id, $amount, $description = null, $mobile = null, $email = null) {
        global $db;
        
        if (!$description) {
            $description = "خرید " . COURSE_NAME;
        }
        
        $callback_url = WEBSITE_URL . '/payment_callback.php';
        
        // Enhanced payment data
        $data = [
            'MerchantID' => $this->merchant_id,
            'Amount' => $amount,
            'Description' => $description,
            'CallbackURL' => $callback_url,
            'Mobile' => $mobile ?: '',
            'Email' => $email ?: '',
            'Currency' => 'IRT', // Iranian Toman
            'OrderId' => $this->generateOrderId($user_id)
        ];
        
        try {
            $response = $this->makeRequest('PaymentRequest.json', $data);
            
            if ($response && $response['Status'] == 100) {
                // Save payment record with ZarinPal authority
                $payment_id = $db->createPayment(
                    $user_id, 
                    $amount, 
                    'zarinpal', 
                    $response['Authority']
                );
                
                // Store additional payment metadata
                $this->storePaymentMetadata($payment_id, [
                    'authority' => $response['Authority'],
                    'order_id' => $data['OrderId'],
                    'mobile' => $mobile,
                    'email' => $email,
                    'sandbox' => $this->sandbox
                ]);
                
                $payment_url = $this->gateway_url . $response['Authority'];
                
                // Log successful payment creation
                error_log("ZarinPal payment created: Payment ID $payment_id, Authority: {$response['Authority']}");
                
                return [
                    'success' => true,
                    'payment_url' => $payment_url,
                    'authority' => $response['Authority'],
                    'payment_id' => $payment_id
                ];
            }
            
            // Handle ZarinPal errors
            $error_message = $this->getZarinPalError($response['Status'] ?? -1);
            error_log("ZarinPal payment creation failed: " . $error_message);
            
            return [
                'success' => false,
                'error' => $error_message,
                'status' => $response['Status'] ?? -1
            ];
            
        } catch (Exception $e) {
            error_log("ZarinPal payment exception: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'خطا در اتصال به درگاه پرداخت'
            ];
        }
    }
    
    /**
     * Verify payment with enhanced validation
     */
    public function verifyPayment($authority, $amount) {
        if (empty($authority) || $amount <= 0) {
            return [
                'success' => false,
                'error' => 'اطلاعات پرداخت نامعتبر'
            ];
        }
        
        $data = [
            'MerchantID' => $this->merchant_id,
            'Authority' => $authority,
            'Amount' => $amount
        ];
        
        try {
            $response = $this->makeRequest('PaymentVerification.json', $data);
            
            if ($response && $response['Status'] == 100) {
                return [
                    'success' => true,
                    'ref_id' => $response['RefID'],
                    'status' => $response['Status']
                ];
            }
            
            $error_message = $this->getZarinPalError($response['Status'] ?? -1);
            error_log("ZarinPal verification failed: Authority $authority, Status: " . ($response['Status'] ?? 'unknown'));
            
            return [
                'success' => false,
                'error' => $error_message,
                'status' => $response['Status'] ?? -1
            ];
            
        } catch (Exception $e) {
            error_log("ZarinPal verification exception: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'خطا در تایید پرداخت'
            ];
        }
    }
    
    /**
     * Make HTTP request to ZarinPal API
     */
    private function makeRequest($endpoint, $data) {
        $url = $this->api_url . $endpoint;
        $json_data = json_encode($data);
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $json_data,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Content-Length: ' . strlen($json_data)
            ],
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CONNECTTIMEOUT => 10
        ]);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        curl_close($ch);
        
        if ($curl_error) {
            throw new Exception("cURL Error: $curl_error");
        }
        
        if ($http_code !== 200) {
            throw new Exception("HTTP Error: $http_code");
        }
        
        $result = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("Invalid JSON response");
        }
        
        return $result;
    }
    
    /**
     * Generate unique order ID
     */
    private function generateOrderId($user_id) {
        return 'CT_' . $user_id . '_' . time() . '_' . rand(1000, 9999);
    }
    
    /**
     * Store payment metadata
     */
    private function storePaymentMetadata($payment_id, $metadata) {
        global $db;
        
        try {
            $stmt = $db->connection->prepare("
                UPDATE payments SET 
                admin_notes = ? 
                WHERE id = ?
            ");
            
            $metadata_json = json_encode($metadata);
            $stmt->execute([$metadata_json, $payment_id]);
            
        } catch (Exception $e) {
            error_log("Failed to store payment metadata: " . $e->getMessage());
        }
    }
    
    /**
     * Get ZarinPal error message
     */
    private function getZarinPalError($status) {
        $errors = [
            -1 => 'اطلاعات ارسال شده ناقص است.',
            -2 => 'IP یا مرچنت کد پذیرنده صحیح نیست.',
            -3 => 'مبلغ باید بیشتر از 1,000 ریال باشد.',
            -4 => 'سطح تایید پذیرنده پایین‌تر از سطح نقره‌ای است.',
            -11 => 'درخواست مورد نظر یافت نشد.',
            -12 => 'امکان ویرایش درخواست میسر نمی‌باشد.',
            -21 => 'هیچ نوع عملیات مالی برای این تراکنش یافت نشد.',
            -22 => 'تراکنش ناموفق می‌باشد.',
            -33 => 'رقم تراکنش با رقم پرداخت شده مطابقت ندارد.',
            -34 => 'سقف تقسیم تراکنش از لحاظ تعداد یا مبلغ عبور کرده است.',
            -40 => 'اجازه دسترسی به متد مربوطه وجود ندارد.',
            -41 => 'اطلاعات ارسال شده مربوط به AdditionalData غیرمعتبر است.',
            -42 => 'مدت زمان معتبر طول عمر شناسه پرداخت باید بین 30 دقیقه تا 45 روز باشد.',
            -54 => 'درخواست مورد نظر آرشیو شده است.',
            101 => 'تراکنش قبلاً وریفای شده است.'
        ];
        
        return $errors[$status] ?? "خطای ناشناخته (کد: $status)";
    }
    
    /**
     * Get payment status description
     */
    public function getStatusDescription($status) {
        $descriptions = [
            -1 => 'در انتظار پردازش',
            -2 => 'خطای پرداخت',
            1 => 'پرداخت موفق',
            2 => 'خطای داخلی',
            3 => 'خطای امنیتی',
            4 => 'خطای اعتبارسنجی'
        ];
        
        return $descriptions[$status] ?? 'وضعیت نامشخص';
    }
}

/**
 * Enhanced Card-to-Card Payment Handler
 */
class CardPayment {
    
    /**
     * Process card payment confirmation with receipt validation
     */
    public static function confirmPayment($user_id, $receipt_data, $amount) {
        global $db, $telegram;
        
        // Create payment record
        $payment_id = $db->createPayment($user_id, $amount, 'card_to_card');
        
        // Store receipt information
        self::storeReceiptData($payment_id, $receipt_data);
        
        // Get user info
        $user = $db->getUserInfo($user_id);
        
        // Notify admins about new payment
        self::notifyAdminsNewPayment($payment_id, $user, $amount, $receipt_data);
        
        // Update payment status to reviewing
        $db->updatePaymentStatus($payment_id, 'reviewing');
        
        return $payment_id;
    }
    
    /**
     * Store receipt data securely
     */
    private static function storeReceiptData($payment_id, $receipt_data) {
        global $db;
        
        try {
            $receipt_metadata = [
                'receipt_image' => $receipt_data['image'] ?? null,
                'upload_time' => date('Y-m-d H:i:s'),
                'file_size' => $receipt_data['file_size'] ?? 0,
                'file_type' => $receipt_data['file_type'] ?? 'unknown'
            ];
            
            $stmt = $db->connection->prepare("
                UPDATE payments SET 
                receipt_image = ?,
                admin_notes = ? 
                WHERE id = ?
            ");
            
            $stmt->execute([
                $receipt_data['image'],
                json_encode($receipt_metadata),
                $payment_id
            ]);
            
        } catch (Exception $e) {
            error_log("Failed to store receipt data: " . $e->getMessage());
        }
    }
    
    /**
     * Notify admins about new payment
     */
    private static function notifyAdminsNewPayment($payment_id, $user, $amount, $receipt_data) {
        global $telegram;
        
        $admin_message = "💰 <b>پرداخت کارت به کارت جدید</b>\n\n";
        $admin_message .= "👤 <b>کاربر:</b> {$user['first_name']}";
        
        if ($user['username']) {
            $admin_message .= " (@{$user['username']})";
        }
        
        $admin_message .= "\n💵 <b>مبلغ:</b> " . number_format($amount) . " تومان";
        $admin_message .= "\n🆔 <b>کد پرداخت:</b> <code>$payment_id</code>";
        $admin_message .= "\n📱 <b>شماره:</b> " . ($user['mobile_number'] ?: 'ثبت نشده');
        $admin_message .= "\n⏰ <b>زمان:</b> " . date('Y/m/d H:i');
        $admin_message .= "\n\n✅ برای تایید: /approve_$payment_id";
        $admin_message .= "\n❌ برای رد: /reject_$payment_id";
        
        // Send to admin if chat ID is configured
        if (defined('ADMIN_CHAT_ID') && ADMIN_CHAT_ID) {
            if (isset($receipt_data['image'])) {
                $telegram->sendPhoto(ADMIN_CHAT_ID, $receipt_data['image'], $admin_message);
            } else {
                $telegram->sendMessage(ADMIN_CHAT_ID, $admin_message);
            }
        }
        
        // Log the notification
        error_log("Admin notified about payment: $payment_id");
    }
    
    /**
     * Approve payment by admin with license creation
     */
    public static function approvePayment($payment_id, $approved_by = 'admin') {
        global $db, $telegram, $spotPlayer;
        
        try {
            // Get payment info
            $payment = $db->getPaymentInfo($payment_id);
            
            if (!$payment) {
                return [
                    'success' => false,
                    'error' => 'پرداخت یافت نشد'
                ];
            }
            
            if ($payment['status'] === 'completed') {
                return [
                    'success' => false,
                    'error' => 'این پرداخت قبلاً تایید شده است'
                ];
            }
            
            // Update payment status
            $db->updatePaymentStatus($payment_id, 'completed', null, $approved_by);
            
            // Create SpotPlayer license
            $license_result = $spotPlayer->createLicense(
                $payment['first_name'], 
                $payment['mobile_number'] ?: '09000000000'
            );
            
            if ($license_result['success']) {
                // Save license to user
                $db->setUserLicense($payment['user_id'], $license_result['license_key']);
                
                // Store license in SpotPlayer table
                $db->createSpotPlayerLicense($payment['user_id'], $payment_id, $license_result);
                
                // Send license to user
                $message = Messages::getLicenseDeliveryMessage(
                    $license_result['license_key'],
                    $license_result['download_url']
                );
                
                $telegram->sendMessage($payment['chat_id'], $message);
                
                // Send success notification to admin
                if (defined('ADMIN_CHAT_ID') && ADMIN_CHAT_ID) {
                    $admin_success = "✅ <b>پرداخت تایید شد</b>\n\n";
                    $admin_success .= "🆔 کد: <code>$payment_id</code>\n";
                    $admin_success .= "👤 کاربر: {$payment['first_name']}\n";
                    $admin_success .= "🎫 لایسنس ارسال شد";
                    
                    $telegram->sendMessage(ADMIN_CHAT_ID, $admin_success);
                }
                
                error_log("Payment approved and license created: Payment ID $payment_id");
                
                return [
                    'success' => true,
                    'license_created' => true,
                    'license_key' => $license_result['license_key']
                ];
                
            } else {
                // License creation failed, but payment is still approved
                error_log("License creation failed for payment $payment_id: " . $license_result['error']);
                
                // Notify user about manual license creation
                $message = "✅ <b>پرداخت شما تایید شد!</b>\n\n";
                $message .= "⏳ لایسنس شما در حال آماده‌سازی است و تا چند ساعت آینده ارسال خواهد شد.\n\n";
                $message .= "📞 برای پیگیری با پشتیبانی تماس بگیرید:\n" . SUPPORT_USERNAME;
                
                $telegram->sendMessage($payment['chat_id'], $message);
                
                return [
                    'success' => true,
                    'license_created' => false,
                    'error' => $license_result['error']
                ];
            }
            
        } catch (Exception $e) {
            error_log("Payment approval failed: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'خطا در تایید پرداخت'
            ];
        }
    }
    
    /**
     * Reject payment by admin
     */
    public static function rejectPayment($payment_id, $reason = '', $rejected_by = 'admin') {
        global $db, $telegram;
        
        try {
            // Get payment info
            $payment = $db->getPaymentInfo($payment_id);
            
            if (!$payment) {
                return [
                    'success' => false,
                    'error' => 'پرداخت یافت نشد'
                ];
            }
            
            // Update payment status
            $db->updatePaymentStatus($payment_id, 'failed', null, $rejected_by);
            
            // Add rejection reason to notes
            if ($reason) {
                $notes = json_decode($payment['admin_notes'] ?: '{}', true);
                $notes['rejection_reason'] = $reason;
                $notes['rejected_at'] = date('Y-m-d H:i:s');
                
                $stmt = $db->connection->prepare("
                    UPDATE payments SET admin_notes = ? WHERE id = ?
                ");
                $stmt->execute([json_encode($notes), $payment_id]);
            }
            
            // Notify user about rejection
            $message = "❌ <b>پرداخت رد شد</b>\n\n";
            $message .= "🆔 <b>کد پرداخت:</b> <code>$payment_id</code>\n\n";
            
            if ($reason) {
                $message .= "📝 <b>دلیل:</b> $reason\n\n";
            }
            
            $message .= "💡 لطفاً رسید واضح‌تری ارسال کنید یا با پشتیبانی تماس بگیرید:\n";
            $message .= SUPPORT_USERNAME;
            
            $telegram->sendMessage($payment['chat_id'], $message);
            
            error_log("Payment rejected: Payment ID $payment_id, Reason: $reason");
            
            return ['success' => true];
            
        } catch (Exception $e) {
            error_log("Payment rejection failed: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'خطا در رد پرداخت'
            ];
        }
    }
    
    /**
     * Get payment statistics
     */
    public static function getPaymentStats() {
        global $db;
        
        try {
            $stats = [];
            
            // Total payments by method
            $stmt = $db->connection->query("
                SELECT payment_method, COUNT(*) as count, SUM(amount) as total
                FROM payments 
                WHERE status = 'completed'
                GROUP BY payment_method
            ");
            $stats['by_method'] = $stmt->fetchAll();
            
            // Daily revenue (last 30 days)
            $stmt = $db->connection->query("
                SELECT DATE(created_at) as date, COUNT(*) as count, SUM(amount) as total
                FROM payments 
                WHERE status = 'completed' 
                AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
                GROUP BY DATE(created_at)
                ORDER BY date DESC
            ");
            $stats['daily_revenue'] = $stmt->fetchAll();
            
            // Pending payments
            $stmt = $db->connection->query("
                SELECT COUNT(*) as count FROM payments WHERE status = 'pending'
            ");
            $stats['pending_count'] = $stmt->fetch()['count'];
            
            return $stats;
            
        } catch (Exception $e) {
            error_log("Failed to get payment stats: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Validate receipt image
     */
    public static function validateReceiptImage($file_data) {
        $allowed_types = ['image/jpeg', 'image/png', 'image/jpg'];
        $max_size = 5 * 1024 * 1024; // 5MB
        
        if (!in_array($file_data['type'], $allowed_types)) {
            return [
                'valid' => false,
                'error' => 'فرمت فایل باید JPG یا PNG باشد'
            ];
        }
        
        if ($file_data['size'] > $max_size) {
            return [
                'valid' => false,
                'error' => 'حجم فایل نباید بیشتر از 5 مگابایت باشد'
            ];
        }
        
        return ['valid' => true];
    }
}

/**
 * Payment Utilities
 */
class PaymentUtils {
    
    /**
     * Format amount for display
     */
    public static function formatAmount($amount, $currency = 'تومان') {
        return number_format($amount) . ' ' . $currency;
    }
    
    /**
     * Generate payment reference
     */
    public static function generatePaymentReference($user_id, $payment_method) {
        $prefix = $payment_method === 'zarinpal' ? 'ZP' : 'CT';
        return $prefix . '_' . $user_id . '_' . time() . '_' . rand(100, 999);
    }
    
    /**
     * Calculate discount amount
     */
    public static function calculateDiscount($amount, $discount_percent) {
        return $amount * ($discount_percent / 100);
    }
    
    /**
     * Get final amount after discount
     */
    public static function getFinalAmount($amount, $discount_percent = 0) {
        $discount = self::calculateDiscount($amount, $discount_percent);
        return $amount - $discount;
    }
}
?>