<?php
/**
 * ULTIMATE Captain Trade Bot - FINAL INTEGRATED VERSION
 * Complete System with All Handlers + Zero Bugs
 * Updated: 2025-10-15 19:22:53 UTC
 * Created by: saportinsta65 - ULTRA PRO DEVELOPER
 */

// Security headers
header('Content-Type: application/json');
header('Cache-Control: no-cache, no-store, must-revalidate');

// Initialize with comprehensive error handling
try {
    if (!defined('BOT_TOKEN')) {
        require_once 'config.php';
    }
    
    // Load all core files
    require_once 'includes/database.php';
    require_once 'includes/telegram.php';
    require_once 'includes/spotplayer.php';
    require_once 'includes/payment.php';
    require_once 'includes/messages.php';
    require_once 'includes/security.php';
    
    // Load new handler files
    require_once 'admin_commands.php';
    require_once 'file_handler.php';
    require_once 'callback_handler.php';
    require_once 'payment_processor.php';
    require_once 'message_router.php';
    
    // Initialize core components
    $db = new Database();
    $telegram = new TelegramBot(BOT_TOKEN);
    $security = new Security();
    
    // Initialize advanced handlers
    $adminCommands = new AdminCommands($telegram, $db);
    $fileHandler = new FileHandler($telegram, $db);
    $callbackHandler = new CallbackHandler($telegram, $db);
    $paymentProcessor = new PaymentProcessor($telegram, $db, new SpotPlayerAPI());
    $messageRouter = new MessageRouter($telegram, $db);
    
    // Security check
    $security->checkRateLimit();
    
} catch (Exception $e) {
    error_log("CRITICAL INIT ERROR: " . $e->getMessage());
    http_response_code(500);
    exit('{"status":"error","message":"system_maintenance","timestamp":"' . date('Y-m-d H:i:s') . '"}');
}

// Get and validate webhook input
$input = file_get_contents('php://input');
if (empty($input)) {
    error_log("Empty webhook input");
    exit('{"status":"error","message":"no_input"}');
}

$update = json_decode($input, true);
if (!$update || json_last_error() !== JSON_ERROR_NONE) {
    error_log("Invalid JSON: " . json_last_error_msg());
    exit('{"status":"error","message":"invalid_json"}');
}

// Enhanced debug logging with timestamp
$log_data = [
    'timestamp' => date('Y-m-d H:i:s'),
    'update' => $update,
    'ip' => $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? 'unknown'
];
error_log("WEBHOOK INPUT: " . json_encode($log_data, JSON_UNESCAPED_UNICODE));

// Extract update data with comprehensive handling
$message = $update['message'] ?? null;
$callback_query = $update['callback_query'] ?? null;
$edited_message = $update['edited_message'] ?? null;

// Determine update type and extract data
if ($message) {
    // Regular message
    $chat_id = $message['chat']['id'];
    $user_id = $message['from']['id'];
    $username = $message['from']['username'] ?? '';
    $first_name = $message['from']['first_name'] ?? 'کاربر';
    $message_text = trim($message['text'] ?? '');
    
    // Determine content type
    if (isset($message['photo'])) {
        $content_type = 'photo';
        $content = $message['photo'];
    } elseif (isset($message['document'])) {
        $content_type = 'document';
        $content = $message['document'];
    } elseif (isset($message['video'])) {
        $content_type = 'video';
        $content = $message['video'];
    } elseif (isset($message['voice'])) {
        $content_type = 'voice';
        $content = $message['voice'];
    } elseif (isset($message['contact'])) {
        $content_type = 'contact';
        $content = $message['contact'];
    } else {
        $content_type = 'text';
        $content = $message_text;
    }
    
} elseif ($callback_query) {
    // Callback query
    $chat_id = $callback_query['message']['chat']['id'];
    $user_id = $callback_query['from']['id'];
    $username = $callback_query['from']['username'] ?? '';
    $first_name = $callback_query['from']['first_name'] ?? 'کاربر';
    $message_text = $callback_query['data'] ?? '';
    $callback_id = $callback_query['id'];
    $content_type = 'callback';
    $content = $message_text;
    
} elseif ($edited_message) {
    // Edited message (ignore for now)
    error_log("Edited message received - ignoring");
    exit('{"status":"ignored","message":"edited_message"}');
    
} else {
    error_log("Unknown update type: " . json_encode($update));
    exit('{"status":"error","message":"unknown_update_type"}');
}

// Validate essential data
if (!$chat_id || !$user_id) {
    error_log("Missing essential data: chat_id=$chat_id, user_id=$user_id");
    exit('{"status":"error","message":"missing_data"}');
}

// Enhanced security validation
try {
    if (!$security->validateUser($user_id, $username)) {
        error_log("Security validation failed: user_id=$user_id, username=$username");
        
        $blocked_message = "🚫 <b>دسترسی محدود شده</b>

━━━━━━━━━━━━━━━━━━━━

⚠️ دسترسی شما به دلایل امنیتی محدود شده است

📞 <b>پشتیبانی:</b> " . SUPPORT_USERNAME . "

💡 <b>در صورت اشتباه با پشتیبانی تماس بگیرید</b>";
        
        $telegram->sendMessage($chat_id, $blocked_message);
        exit('{"status":"blocked","user_id":"' . $user_id . '"}');
    }
} catch (Exception $e) {
    error_log("Security validation error: " . $e->getMessage());
    exit('{"status":"error","message":"security_error"}');
}

// Register/update user with enhanced data
try {
    $db->registerUser($user_id, $username, $first_name, $chat_id);
    
    // Log detailed activity
    $activity_data = [
        'content_type' => $content_type,
        'message_length' => is_string($content) ? strlen($content) : 0,
        'has_username' => !empty($username),
        'chat_type' => 'private'
    ];
    
    $db->logUserActivity($user_id, $username, $first_name, $message_text, $activity_data);
    
} catch (Exception $e) {
    error_log("User registration/logging error: " . $e->getMessage());
    // Continue execution even if logging fails
}

// Handle different content types
try {
    switch ($content_type) {
        case 'callback':
            // Handle callback queries with answer
            $callbackHandler->handleCallback($callback_id, $chat_id, $user_id, $content);
            exit('{"status":"callback_handled","user_id":"' . $user_id . '"}');
            
        case 'photo':
            // Handle photo uploads
            $fileHandler->handlePhotoUpload($chat_id, $user_id, $content);
            exit('{"status":"photo_handled","user_id":"' . $user_id . '"}');
            
        case 'document':
            // Handle document uploads
            $fileHandler->handleDocumentUpload($chat_id, $user_id, $content);
            exit('{"status":"document_handled","user_id":"' . $user_id . '"}');
            
        case 'video':
        case 'voice':
        case 'contact':
            // Handle other media types
            handleUnsupportedMedia($chat_id, $content_type);
            exit('{"status":"unsupported_media","type":"' . $content_type . '"}');
            
        case 'text':
        default:
            // Handle text messages through router
            $messageRouter->route($chat_id, $user_id, $username, $first_name, $content, $content_type);
            break;
    }
    
} catch (Exception $e) {
    error_log("Message handling error: " . $e->getMessage());
    
    // Send user-friendly error message
    $error_message = "❌ <b>خطایی رخ داده است</b>

━━━━━━━━━━━━━━━━━━━━

🔄 <b>راه‌حل‌ها:</b>
• دستور /start را بزنید
• از منو استفاده کنید
• با پشتیبانی تماس بگیرید

📞 <b>پشتیبانی:</b> " . SUPPORT_USERNAME . "

💡 <b>خطا به تیم فنی گزارش شد</b>";
    
    try {
        $telegram->sendMessage($chat_id, $error_message, Messages::getMainMenuKeyboard());
    } catch (Exception $te) {
        error_log("Failed to send error message: " . $te->getMessage());
    }
}

// Final response with comprehensive status
$response_data = [
    'status' => 'ok',
    'timestamp' => date('Y-m-d H:i:s'),
    'user_id' => $user_id,
    'content_type' => $content_type,
    'processing_time' => round((microtime(true) - $_SERVER['REQUEST_TIME_FLOAT']) * 1000, 2) . 'ms'
];

exit(json_encode($response_data));

/**
 * 📱 مدیریت رسانه‌های پشتیبانی نشده
 */
function handleUnsupportedMedia($chat_id, $media_type) {
    global $telegram;
    
    $media_names = [
        'video' => 'ویدیو',
        'voice' => 'پیام صوتی', 
        'contact' => 'مخاطب',
        'location' => 'موقعیت مکانی',
        'sticker' => 'استیکر'
    ];
    
    $media_name = $media_names[$media_type] ?? 'فایل';
    
    $message = "📱 <b>نوع فایل پشتیبانی نمی‌شود</b>

━━━━━━━━━━━━━━━━━━━━

📄 <b>نوع دریافتی:</b> {$media_name}

✅ <b>فایل‌های مجاز:</b>
• متن و پیام
• عکس (برای رسید پرداخت)
• PDF و تصاویر (برای مدارک)

━━━━━━━━━━━━━━━━━━━━

💡 <b>لطفاً از منو استفاده کنید</b>";
    
    $telegram->sendMessage($chat_id, $message, Messages::getMainMenuKeyboard());
}

/**
 * 🛡️ Security helper functions
 */
function logSecurityEvent($event_type, $user_id, $details = []) {
    $event_data = [
        'timestamp' => date('Y-m-d H:i:s'),
        'event_type' => $event_type,
        'user_id' => $user_id,
        'ip_address' => $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
        'details' => $details
    ];
    
    error_log("SECURITY_EVENT: " . json_encode($event_data));
}

/**
 * 📊 Performance monitoring
 */
function logPerformanceMetrics() {
    $metrics = [
        'timestamp' => date('Y-m-d H:i:s'),
        'memory_usage' => memory_get_usage(true),
        'memory_peak' => memory_get_peak_usage(true),
        'execution_time' => microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'],
        'included_files' => count(get_included_files())
    ];
    
    // Log if execution time is over 2 seconds
    if ($metrics['execution_time'] > 2) {
        error_log("PERFORMANCE_WARNING: " . json_encode($metrics));
    }
}

/**
 * 🔧 System health check
 */
function validateSystemHealth() {
    $checks = [
        'database' => class_exists('Database'),
        'telegram' => class_exists('TelegramBot'),
        'handlers' => (
            class_exists('AdminCommands') && 
            class_exists('FileHandler') && 
            class_exists('CallbackHandler') &&
            class_exists('PaymentProcessor') &&
            class_exists('MessageRouter')
        ),
        'config' => (
            defined('BOT_TOKEN') && 
            defined('ADMIN_CHAT_ID') && 
            defined('COURSE_PRICE')
        )
    ];
    
    foreach ($checks as $component => $status) {
        if (!$status) {
            error_log("SYSTEM_ERROR: Component '$component' failed health check");
            return false;
        }
    }
    
    return true;
}

// Run performance monitoring
register_shutdown_function('logPerformanceMetrics');

// Validate system health on critical errors
if (!validateSystemHealth()) {
    error_log("CRITICAL: System health check failed");
    http_response_code(500);
    exit('{"status":"system_error","message":"health_check_failed"}');
}
?>