<?php
/**
 * ═══════════════════════════════════════════════════
 * 💾 Database Helper Functions
 * ═══════════════════════════════════════════════════
 */

// ─────────────────────────────────────────────────
// 👤 User Management
// ─────────────────────────────────────────────────

function saveUser($user_id, $username = '', $first_name = '') {
    $users = file_exists(USERS_FILE) ? file(USERS_FILE, FILE_IGNORE_NEW_LINES) : [];
    
    foreach($users as $u) {
        if(explode('|', $u)[0] == $user_id) return false;
    }
    
    $data = $user_id . '|' . time() . '|' . $username . '|' . $first_name . PHP_EOL;
    file_put_contents(USERS_FILE, $data, FILE_APPEND);
    
    logActivity("New user registered: $user_id ($first_name)");
    return true;
}

function getUserData($user_id) {
    if(!file_exists(USERS_FILE)) return null;
    
    $users = file(USERS_FILE, FILE_IGNORE_NEW_LINES);
    foreach($users as $u) {
        $parts = explode('|', $u);
        if($parts[0] == $user_id) {
            return [
                'id' => $parts[0],
                'registered_at' => $parts[1],
                'username' => $parts[2] ?? '',
                'first_name' => $parts[3] ?? ''
            ];
        }
    }
    return null;
}

function isUserInChannel($user_id) {
    $res = apiRequest('getChatMember', [
        'chat_id' => CHANNEL_USERNAME,
        'user_id' => $user_id
    ]);
    
    if(isset($res['result']['status'])) {
        $status = $res['result']['status'];
        return in_array($status, ['member', 'administrator', 'creator']);
    }
    return false;
}

function getAllUsers() {
    if(!file_exists(USERS_FILE)) return [];
    return file(USERS_FILE, FILE_IGNORE_NEW_LINES);
}

function getUserCount($period = 'all') {
    $users = getAllUsers();
    if($period == 'all') return count($users);
    
    $now = time();
    $count = 0;
    $seconds = [
        'today' => 86400,
        'week' => 7 * 86400,
        'month' => 30 * 86400
    ];
    
    foreach($users as $u) {
        $reg_time = explode('|', $u)[1];
        if($now - $reg_time <= ($seconds[$period] ?? 0)) {
            $count++;
        }
    }
    
    return $count;
}

// ─────────────────────────────────────────────────
// 📦 Order Management
// ─────────────────────────────────────────────────

function loadOrders() {
    return file_exists(ORDERS_FILE) 
        ? json_decode(file_get_contents(ORDERS_FILE), true) 
        : [];
}

function saveOrders($orders) {
    file_put_contents(ORDERS_FILE, json_encode($orders, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function createOrder($user_id, $service, $amount, $payment_method = 'manual') {
    $orders = loadOrders();
    
    $order_id = 'ORD-' . date('ymd') . '-' . strtoupper(substr(md5(time() . $user_id), 0, 6));
    
    $orders[$order_id] = [
        'user_id' => $user_id,
        'service' => $service,
        'amount' => $amount,
        'payment_method' => $payment_method,
        'status' => 'pending', // pending, processing, completed, cancelled
        'created_at' => time(),
        'updated_at' => time(),
        'notes' => []
    ];
    
    saveOrders($orders);
    logActivity("New order created: $order_id for user $user_id");
    
    return $order_id;
}

function getOrder($order_id) {
    $orders = loadOrders();
    return $orders[$order_id] ?? null;
}

function getUserOrders($user_id) {
    $orders = loadOrders();
    return array_filter($orders, function($order) use ($user_id) {
        return $order['user_id'] == $user_id;
    });
}

function updateOrderStatus($order_id, $status, $note = '') {
    $orders = loadOrders();
    
    if(!isset($orders[$order_id])) return false;
    
    $orders[$order_id]['status'] = $status;
    $orders[$order_id]['updated_at'] = time();
    
    if($note) {
        $orders[$order_id]['notes'][] = [
            'text' => $note,
            'time' => time()
        ];
    }
    
    saveOrders($orders);
    logActivity("Order $order_id status updated to: $status");
    
    return true;
}

// ─────────────────────────────────────────────────
// 💰 Payment Management
// ─────────────────────────────────────────────────

function loadPayments() {
    return file_exists(PAYMENTS_FILE) 
        ? json_decode(file_get_contents(PAYMENTS_FILE), true) 
        : [];
}

function savePayments($payments) {
    file_put_contents(PAYMENTS_FILE, json_encode($payments, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

function createPayment($order_id, $amount, $method = 'online') {
    $payments = loadPayments();
    
    $payment_id = 'PAY-' . time() . '-' . rand(1000, 9999);
    
    $payments[$payment_id] = [
        'order_id' => $order_id,
        'amount' => $amount,
        'method' => $method,
        'status' => 'pending', // pending, completed, failed
        'authority' => null,
        'ref_id' => null,
        'created_at' => time(),
        'verified_at' => null
    ];
    
    savePayments($payments);
    return $payment_id;
}

function getPayment($payment_id) {
    $payments = loadPayments();
    return $payments[$payment_id] ?? null;
}

function updatePayment($payment_id, $data) {
    $payments = loadPayments();
    
    if(!isset($payments[$payment_id])) return false;
    
    foreach($data as $key => $value) {
        $payments[$payment_id][$key] = $value;
    }
    
    savePayments($payments);
    return true;
}

// ─────────────────────────────────────────────────
// 📊 Statistics
// ─────────────────────────────────────────────────

function getStats() {
    $users = getAllUsers();
    $orders = loadOrders();
    
    $stats = [
        'total_users' => count($users),
        'users_today' => getUserCount('today'),
        'users_week' => getUserCount('week'),
        'users_month' => getUserCount('month'),
        'total_orders' => count($orders),
        'orders_today' => 0,
        'orders_week' => 0,
        'total_revenue' => 0,
        'completed_orders' => 0,
        'pending_orders' => 0
    ];
    
    $now = time();
    
    foreach($orders as $order) {
        $stats['total_revenue'] += $order['amount'];
        
        if($order['status'] == 'completed') {
            $stats['completed_orders']++;
        } elseif($order['status'] == 'pending') {
            $stats['pending_orders']++;
        }
        
        if($now - $order['created_at'] <= 86400) {
            $stats['orders_today']++;
        }
        if($now - $order['created_at'] <= 7 * 86400) {
            $stats['orders_week']++;
        }
    }
    
    $stats['conversion_rate'] = $stats['total_users'] > 0 
        ? round(($stats['completed_orders'] / $stats['total_users']) * 100, 2) 
        : 0;
    
    return $stats;
}
?>