<?php
/**
 * ═══════════════════════════════════════════════════
 * ❓ FAQ System Module
 * ═══════════════════════════════════════════════════
 */

/**
 * بارگذاری سوالات متداول
 */
function loadFAQs() {
    if(!file_exists(FAQ_FILE)) {
        // ایجاد FAQ پیش‌فرض
        $default_faqs = [
            [
                'question' => 'آیا خدمات شما قانونی است؟',
                'answer' => 'بله، تمام خدمات ما با رعایت قوانین و فقط برای اهداف قانونی ارائه می‌شود. استفاده از خدمات برای فعالیت‌های غیرقانونی اکیداً ممنوع است.',
                'category' => 'general'
            ],
            [
                'question' => 'چقدر طول می‌کشد تا کار من انجام شود؟',
                'answer' => 'بسته به نوع سرویس، زمان انجام کار بین ۲ تا ۷۲ ساعت متغیر است. برای اطلاعات دقیق‌تر، توضیحات هر سرویس را مطالعه کنید.',
                'category' => 'general'
            ],
            [
                'question' => 'آیا اطلاعات من محفوظ است؟',
                'answer' => 'بله، ما به حریم خصوصی شما احترام می‌گذاریم. تمام اطلاعات شما کاملاً محرمانه است و هیچ‌گاه به اشتراک گذاشته نمی‌شود.',
                'category' => 'security'
            ],
            [
                'question' => 'اگر کار انجام نشود چه می‌شود؟',
                'answer' => 'ما ۱۰۰٪ تضمین می‌دهیم که در صورت عدم انجام کار، کل مبلغ پرداختی شما بازگردانده می‌شود.',
                'category' => 'payment'
            ],
            [
                'question' => 'چطور می‌توانم پرداخت کنم؟',
                'answer' => 'شما می‌توانید از دو روش پرداخت استفاده کنید:\n\n1️⃣ پرداخت آنلاین از طریق درگاه زرین‌پال\n2️⃣ کارت به کارت و ارسال رسید',
                'category' => 'payment'
            ],
            [
                'question' => 'آیا می‌توانم با پشتیبانی صحبت کنم؟',
                'answer' => 'بله، تیم پشتیبانی ما ۲۴/۷ پاسخگوی شماست. از منوی اصلی گزینه «پشتیبانی» را انتخاب کنید.',
                'category' => 'support'
            ],
            [
                'question' => 'آیا تخفیف دارید؟',
                'answer' => 'بله! برای اولین خرید ۲۰٪ تخفیف دریافت می‌کنید. همچنین با دعوت دوستان می‌توانید کمیسیون کسب کنید.',
                'category' => 'payment'
            ],
            [
                'question' => 'چگونه می‌توانم وضعیت سفارشم را ببینم؟',
                'answer' => 'از منوی اصلی، گزینه «سفارشات من» را انتخاب کنید. آنجا می‌توانید وضعیت و جزئیات تمام سفارشات خود را مشاهده کنید.',
                'category' => 'general'
            ]
        ];
        
        file_put_contents(FAQ_FILE, json_encode($default_faqs, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        return $default_faqs;
    }
    
    return json_decode(file_get_contents(FAQ_FILE), true);
}

/**
 * ذخیره سوالات متداول
 */
function saveFAQs($faqs) {
    file_put_contents(FAQ_FILE, json_encode($faqs, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

/**
 * نمایش دسته‌بندی سوالات
 */
function showFAQCategories($chat_id) {
    $categories = [
        'general' => '📌 سوالات عمومی',
        'security' => '🔒 امنیت و حریم خصوصی',
        'payment' => '💰 پرداخت و قیمت‌ها',
        'support' => '💬 پشتیبانی'
    ];
    
    $keyboard = [];
    foreach($categories as $key => $label) {
        $keyboard[] = [['text' => $label, 'callback_data' => "faq_cat_$key"]];
    }
    $keyboard[] = [['text' => '📋 مشاهده همه سوالات', 'callback_data' => 'faq_all']];
    $keyboard[] = [['text' => '🔙 بازگشت', 'callback_data' => 'back_main']];
    
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => "
❓ <b>سوالات متداول (FAQ)</b>

دسته‌بندی موردنظر خود را انتخاب کنید:
",
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode(['inline_keyboard' => $keyboard])
    ]);
}

/**
 * نمایش سوالات یک دسته
 */
function showFAQByCategory($chat_id, $category) {
    $faqs = loadFAQs();
    $filtered = array_filter($faqs, function($faq) use ($category) {
        return $faq['category'] == $category;
    });
    
    if(empty($filtered)) {
        apiRequest('sendMessage', [
            'chat_id' => $chat_id,
            'text' => "هیچ سوالی در این دسته یافت نشد."
        ]);
        return;
    }
    
    $keyboard = [];
    foreach($filtered as $index => $faq) {
        $keyboard[] = [['text' => $faq['question'], 'callback_data' => "faq_$index"]];
    }
    $keyboard[] = [['text' => '🔙 بازگشت', 'callback_data' => 'faq_main']];
    
    $category_names = [
        'general' => '📌 سوالات عمومی',
        'security' => '🔒 امنیت و حریم خصوصی',
        'payment' => '💰 پرداخت و قیمت‌ها',
        'support' => '💬 پشتیبانی'
    ];
    
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => "
{$category_names[$category]}

سوال موردنظر خود را انتخاب کنید:
",
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode(['inline_keyboard' => $keyboard])
    ]);
}

/**
 * نمایش همه سوالات
 */
function showAllFAQs($chat_id) {
    $faqs = loadFAQs();
    
    if(empty($faqs)) {
        apiRequest('sendMessage', [
            'chat_id' => $chat_id,
            'text' => "هنوز سوالی ثبت نشده است."
        ]);
        return;
    }
    
    $keyboard = [];
    foreach($faqs as $index => $faq) {
        $keyboard[] = [['text' => $faq['question'], 'callback_data' => "faq_$index"]];
    }
    $keyboard[] = [['text' => '🔙 بازگشت', 'callback_data' => 'faq_main']];
    
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => "
❓ <b>تمام سوالات متداول</b>

سوال موردنظر خود را انتخاب کنید:
",
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode(['inline_keyboard' => $keyboard])
    ]);
}

/**
 * نمایش پاسخ یک سوال
 */
function showFAQAnswer($chat_id, $index) {
    $faqs = loadFAQs();
    
    if(!isset($faqs[$index])) {
        apiRequest('sendMessage', [
            'chat_id' => $chat_id,
            'text' => "❌ سوال یافت نشد."
        ]);
        return;
    }
    
    $faq = $faqs[$index];
    
    $message = "
❓ <b>{$faq['question']}</b>

━━━━━━━━━━━━━━━━━━━━
✅ <b>پاسخ:</b>

{$faq['answer']}

━━━━━━━━━━━━━━━━━━━━
💬 سوال دیگری دارید؟ از پشتیبانی بپرسید!
";
    
    $keyboard = [
        [['text' => '◀️ سوال قبلی', 'callback_data' => 'faq_' . ($index - 1)], ['text' => 'سوال بعدی ▶️', 'callback_data' => 'faq_' . ($index + 1)]],
        [['text' => '📋 لیست سوالات', 'callback_data' => 'faq_all']],
        [['text' => '💬 پشتیبانی', 'callback_data' => 'contact_support'], ['text' => '🏠 منوی اصلی', 'callback_data' => 'back_main']]
    ];
    
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => $message,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode(['inline_keyboard' => $keyboard])
    ]);
}

/**
 * جستجوی هوشمند در FAQ
 */
function searchFAQ($query) {
    $faqs = loadFAQs();
    $results = [];
    
    $query = mb_strtolower($query);
    
    foreach($faqs as $index => $faq) {
        $question_lower = mb_strtolower($faq['question']);
        $answer_lower = mb_strtolower($faq['answer']);
        
        if(strpos($question_lower, $query) !== false || strpos($answer_lower, $query) !== false) {
            $results[$index] = $faq;
        }
    }
    
    return $results;
}

/**
 * افزودن سوال جدید (برای ادمین)
 */
function addFAQ($question, $answer, $category = 'general') {
    $faqs = loadFAQs();
    
    $faqs[] = [
        'question' => $question,
        'answer' => $answer,
        'category' => $category
    ];
    
    saveFAQs($faqs);
    logActivity("New FAQ added: $question");
    
    return true;
}

/**
 * حذف سوال (برای ادمین)
 */
function deleteFAQ($index) {
    $faqs = loadFAQs();
    
    if(!isset($faqs[$index])) return false;
    
    unset($faqs[$index]);
    $faqs = array_values($faqs); // Re-index array
    
    saveFAQs($faqs);
    logActivity("FAQ deleted: index $index");
    
    return true;
}

/**
 * پنل مدیریت FAQ برای ادمین
 */
function showAdminFAQPanel($chat_id) {
    $faqs = loadFAQs();
    
    $message = "
👨‍💼 <b>پنل مدیریت FAQ</b>

📊 تعداد کل سوالات: " . count($faqs) . "

دسته‌بندی:
📌 عمومی: " . count(array_filter($faqs, fn($f) => $f['category'] == 'general')) . "
🔒 امنیت: " . count(array_filter($faqs, fn($f) => $f['category'] == 'security')) . "
💰 پرداخت: " . count(array_filter($faqs, fn($f) => $f['category'] == 'payment')) . "
💬 پشتیبانی: " . count(array_filter($faqs, fn($f) => $f['category'] == 'support')) . "
";
    
    $keyboard = [
        [['text' => '➕ افزودن سوال جدید', 'callback_data' => 'admin_add_faq']],
        [['text' => '📋 مشاهده همه', 'callback_data' => 'admin_view_faqs'], ['text' => '🗑 حذف سوال', 'callback_data' => 'admin_delete_faq']],
        [['text' => '🔙 بازگشت', 'callback_data' => 'admin_panel']]
    ];
    
    apiRequest('sendMessage', [
        'chat_id' => $chat_id,
        'text' => $message,
        'parse_mode' => 'HTML',
        'reply_markup' => json_encode(['inline_keyboard' => $keyboard])
    ]);
}
?>