/**
 * Admin Panel JavaScript
 * Captain Trade Bot - Professional Admin Interface
 * Updated: 2025-10-15 16:37:09 UTC
 * Created by: saportinsta65
 */

// Global variables
let currentTab = 'dashboard';
let refreshInterval = null;
let notificationSound = null;

// Initialize admin panel
document.addEventListener('DOMContentLoaded', function() {
    initializeAdminPanel();
    startAutoRefresh();
    loadNotificationSound();
});

/**
 * Initialize admin panel functionality
 */
function initializeAdminPanel() {
    // Tab switching
    document.querySelectorAll('[data-tab]').forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            switchTab(this.dataset.tab);
        });
    });
    
    // Load initial data
    loadDashboardData();
    loadRecentActivities();
    
    // Auto-refresh for real-time updates
    setInterval(checkForUpdates, 30000); // Check every 30 seconds
    
    // Initialize tooltips
    if (typeof bootstrap !== 'undefined') {
        var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    }
}

/**
 * Switch between admin panel tabs
 */
function switchTab(tabName) {
    // Hide all tab contents
    document.querySelectorAll('.tab-content').forEach(tab => {
        tab.classList.remove('active');
    });
    
    // Remove active class from all nav links
    document.querySelectorAll('[data-tab]').forEach(link => {
        link.classList.remove('active');
    });
    
    // Show selected tab
    const targetTab = document.getElementById(tabName);
    if (targetTab) {
        targetTab.classList.add('active');
    }
    
    // Add active class to clicked nav link
    const activeLink = document.querySelector(`[data-tab="${tabName}"]`);
    if (activeLink) {
        activeLink.classList.add('active');
    }
    
    currentTab = tabName;
    
    // Load tab-specific data
    loadTabData(tabName);
}

/**
 * Load data for specific tab
 */
function loadTabData(tabName) {
    switch(tabName) {
        case 'dashboard':
            loadDashboardData();
            loadRecentActivities();
            break;
        case 'users':
            loadUsersData();
            break;
        case 'menus':
            loadMenusData();
            break;
        case 'payments':
            loadPaymentsData();
            break;
        case 'analytics':
            loadAnalyticsData();
            break;
        case 'settings':
            loadSettingsData();
            break;
    }
}

/**
 * Load dashboard statistics
 */
function loadDashboardData() {
    fetch('ajax.php?action=get_stats')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updateDashboardStats(data.stats);
            }
        })
        .catch(error => {
            console.error('Error loading dashboard data:', error);
            showNotification('خطا در بارگیری داده‌های داشبورد', 'error');
        });
}

/**
 * Update dashboard statistics
 */
function updateDashboardStats(stats) {
    // Update stat cards
    updateStatCard('total_users', stats.total_users);
    updateStatCard('verified_users', stats.verified_users);
    updateStatCard('total_revenue', formatNumber(stats.total_revenue) + ' تومان');
    updateStatCard('today_registrations', stats.today_registrations);
    updateStatCard('pending_payments', stats.pending_payments);
    updateStatCard('weekly_growth', stats.weekly_growth);
}

/**
 * Update individual stat card
 */
function updateStatCard(cardId, value) {
    const element = document.querySelector(`[data-stat="${cardId}"]`);
    if (element) {
        element.textContent = value;
        
        // Add animation effect
        element.classList.add('stat-updated');
        setTimeout(() => {
            element.classList.remove('stat-updated');
        }, 1000);
    }
}

/**
 * Load recent activities
 */
function loadRecentActivities() {
    fetch('ajax.php?action=get_recent_activities')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updateActivitiesList(data.activities);
            }
        })
        .catch(error => {
            console.error('Error loading activities:', error);
        });
}

/**
 * Update activities list
 */
function updateActivitiesList(activities) {
    const container = document.getElementById('recent-activities');
    if (!container) return;
    
    if (activities.length === 0) {
        container.innerHTML = '<p class="text-muted text-center">فعالیت اخیری یافت نشد.</p>';
        return;
    }
    
    let html = '<div class="list-group list-group-flush">';
    
    activities.forEach(activity => {
        const timeAgo = getTimeAgo(activity.created_at);
        const icon = getActivityIcon(activity.action);
        
        html += `
            <div class="list-group-item border-0 px-0">
                <div class="d-flex align-items-center">
                    <div class="me-3">
                        <i class="bi ${icon} text-primary"></i>
                    </div>
                    <div class="flex-grow-1">
                        <div class="fw-bold text-dark">${escapeHtml(activity.first_name || 'کاربر')}</div>
                        <div class="text-muted small">${escapeHtml(activity.action)}</div>
                    </div>
                    <div class="text-muted small">
                        ${timeAgo}
                    </div>
                </div>
            </div>
        `;
    });
    
    html += '</div>';
    container.innerHTML = html;
}

/**
 * Get icon for activity type
 */
function getActivityIcon(action) {
    if (action.includes('login') || action.includes('ورود')) {
        return 'bi-box-arrow-in-right';
    } else if (action.includes('payment') || action.includes('پرداخت')) {
        return 'bi-credit-card';
    } else if (action.includes('license') || action.includes('لایسنس')) {
        return 'bi-key';
    } else if (action.includes('menu') || action.includes('منو')) {
        return 'bi-menu-button';
    } else {
        return 'bi-activity';
    }
}

/**
 * Load users data
 */
function loadUsersData() {
    fetch('ajax.php?action=get_users')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updateUsersTable(data.users);
            }
        })
        .catch(error => {
            console.error('Error loading users:', error);
            showNotification('خطا در بارگیری کاربران', 'error');
        });
}

/**
 * Update users table
 */
function updateUsersTable(users) {
    const tbody = document.querySelector('#usersTable tbody');
    if (!tbody) return;
    
    if (users.length === 0) {
        tbody.innerHTML = '<tr><td colspan="6" class="text-center text-muted">کاربری یافت نشد.</td></tr>';
        return;
    }
    
    let html = '';
    
    users.forEach(user => {
        const status = user.is_verified ? 
            '<span class="badge bg-success">تایید شده</span>' : 
            '<span class="badge bg-warning">در انتظار تایید</span>';
        
        const createdAt = new Date(user.created_at).toLocaleDateString('fa-IR');
        
        html += `
            <tr>
                <td>${escapeHtml(user.first_name)}</td>
                <td>${user.username ? '@' + escapeHtml(user.username) : '-'}</td>
                <td>${user.mobile_number || '-'}</td>
                <td>${status}</td>
                <td>${createdAt}</td>
                <td>
                    <div class="btn-group btn-group-sm">
                        <button class="btn btn-outline-primary" onclick="viewUser(${user.user_id})" 
                                title="مشاهده جزئیات">
                            <i class="bi bi-eye"></i>
                        </button>
                        <button class="btn btn-outline-${user.is_blocked ? 'success' : 'danger'}" 
                                onclick="toggleUserBlock(${user.user_id}, ${!user.is_blocked})"
                                title="${user.is_blocked ? 'رفع مسدودیت' : 'مسدود کردن'}">
                            <i class="bi bi-${user.is_blocked ? 'unlock' : 'lock'}"></i>
                        </button>
                    </div>
                </td>
            </tr>
        `;
    });
    
    tbody.innerHTML = html;
}

/**
 * Load menus data
 */
function loadMenusData() {
    fetch('ajax.php?action=get_menus')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updateMenusList(data.menus);
            }
        })
        .catch(error => {
            console.error('Error loading menus:', error);
            showNotification('خطا در بارگیری منوها', 'error');
        });
}

/**
 * Update menus list
 */
function updateMenusList(menus) {
    const container = document.getElementById('menus-list');
    if (!container) return;
    
    if (menus.length === 0) {
        container.innerHTML = '<p class="text-muted text-center">منویی یافت نشد.</p>';
        return;
    }
    
    let html = '';
    
    menus.forEach(menu => {
        const isActive = menu.is_active ? 'success' : 'secondary';
        const statusText = menu.is_active ? 'فعال' : 'غیرفعال';
        
        html += `
            <div class="card mb-3">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start">
                        <div class="flex-grow-1">
                            <h5 class="card-title">
                                ${menu.icon} ${escapeHtml(menu.title)}
                                <span class="badge bg-${isActive} ms-2">${statusText}</span>
                            </h5>
                            <p class="card-text text-muted">
                                ${escapeHtml(menu.content.substring(0, 100))}${menu.content.length > 100 ? '...' : ''}
                            </p>
                            <small class="text-muted">
                                <i class="bi bi-eye me-1"></i>
                                ${menu.access_count} بازدید
                            </small>
                        </div>
                        <div class="btn-group">
                            <button class="btn btn-outline-primary btn-sm" onclick="editMenu(${menu.id})"
                                    title="ویرایش">
                                <i class="bi bi-pencil"></i>
                            </button>
                            <button class="btn btn-outline-${menu.is_active ? 'warning' : 'success'} btn-sm" 
                                    onclick="toggleMenu(${menu.id}, ${!menu.is_active})"
                                    title="${menu.is_active ? 'غیرفعال کردن' : 'فعال کردن'}">
                                <i class="bi bi-${menu.is_active ? 'eye-slash' : 'eye'}"></i>
                            </button>
                            <button class="btn btn-outline-danger btn-sm" onclick="deleteMenu(${menu.id})"
                                    title="حذف">
                                <i class="bi bi-trash"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;
    });
    
    container.innerHTML = html;
}

/**
 * Show add menu modal
 */
function showAddMenuModal() {
    const modal = new bootstrap.Modal(document.getElementById('menuModal'));
    
    // Reset form
    document.getElementById('menuForm').reset();
    document.querySelector('#menuModal .modal-title').textContent = 'افزودن منوی جدید';
    document.querySelector('#menuModal [onclick="saveMenu()"]').textContent = 'ذخیره';
    
    // Remove edit mode data
    delete document.getElementById('menuModal').dataset.editId;
    
    modal.show();
}

/**
 * Edit menu
 */
function editMenu(menuId) {
    fetch(`ajax.php?action=get_menu&id=${menuId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const menu = data.menu;
                const modal = new bootstrap.Modal(document.getElementById('menuModal'));
                
                // Fill form with menu data
                document.querySelector('#menuModal [name="title"]').value = menu.title;
                document.querySelector('#menuModal [name="content"]').value = menu.content;
                document.querySelector('#menuModal [name="type"]').value = menu.content_type;
                
                // Set modal title and button
                document.querySelector('#menuModal .modal-title').textContent = 'ویرایش منو';
                document.querySelector('#menuModal [onclick="saveMenu()"]').textContent = 'به‌روزرسانی';
                
                // Store edit ID
                document.getElementById('menuModal').dataset.editId = menuId;
                
                modal.show();
            }
        })
        .catch(error => {
            console.error('Error loading menu:', error);
            showNotification('خطا در بارگیری منو', 'error');
        });
}

/**
 * Save menu (add or update)
 */
function saveMenu() {
    const form = document.getElementById('menuForm');
    const formData = new FormData(form);
    
    const editId = document.getElementById('menuModal').dataset.editId;
    const action = editId ? 'update_menu' : 'add_menu';
    
    formData.append('action', action);
    if (editId) {
        formData.append('id', editId);
    }
    
    fetch('ajax.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification(editId ? 'منو با موفقیت به‌روزرسانی شد' : 'منو با موفقیت اضافه شد', 'success');
            bootstrap.Modal.getInstance(document.getElementById('menuModal')).hide();
            loadMenusData();
        } else {
            showNotification('خطا در ذخیره منو: ' + (data.error || 'خطای نامشخص'), 'error');
        }
    })
    .catch(error => {
        console.error('Error saving menu:', error);
        showNotification('خطا در ذخیره منو', 'error');
    });
}

/**
 * Delete menu
 */
function deleteMenu(menuId) {
    if (!confirm('آیا از حذف این منو اطمینان دارید؟')) {
        return;
    }
    
    const formData = new FormData();
    formData.append('action', 'delete_menu');
    formData.append('id', menuId);
    
    fetch('ajax.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('منو با موفقیت حذف شد', 'success');
            loadMenusData();
        } else {
            showNotification('خطا در حذف منو', 'error');
        }
    })
    .catch(error => {
        console.error('Error deleting menu:', error);
        showNotification('خطا در حذف منو', 'error');
    });
}

/**
 * Toggle menu active status
 */
function toggleMenu(menuId, isActive) {
    const formData = new FormData();
    formData.append('action', 'toggle_menu');
    formData.append('id', menuId);
    formData.append('is_active', isActive ? 1 : 0);
    
    fetch('ajax.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification(`منو ${isActive ? 'فعال' : 'غیرفعال'} شد`, 'success');
            loadMenusData();
        } else {
            showNotification('خطا در تغییر وضعیت منو', 'error');
        }
    })
    .catch(error => {
        console.error('Error toggling menu:', error);
        showNotification('خطا در تغییر وضعیت منو', 'error');
    });
}

/**
 * Load payments data
 */
function loadPaymentsData() {
    fetch('ajax.php?action=get_payments')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                updatePaymentsTable(data.payments);
            }
        })
        .catch(error => {
            console.error('Error loading payments:', error);
            showNotification('خطا در بارگیری پرداخت‌ها', 'error');
        });
}

/**
 * Update payments table
 */
function updatePaymentsTable(payments) {
    const container = document.getElementById('payments-container');
    if (!container) return;
    
    if (payments.length === 0) {
        container.innerHTML = '<p class="text-muted text-center">پرداختی یافت نشد.</p>';
        return;
    }
    
    let html = '<div class="table-responsive"><table class="table table-striped"><thead class="table-dark"><tr>';
    html += '<th>کاربر</th><th>مبلغ</th><th>روش</th><th>وضعیت</th><th>تاریخ</th><th>عملیات</th>';
    html += '</tr></thead><tbody>';
    
    payments.forEach(payment => {
        const statusClass = getPaymentStatusClass(payment.status);
        const statusText = getPaymentStatusText(payment.status);
        const createdAt = new Date(payment.created_at).toLocaleDateString('fa-IR');
        
        html += `
            <tr>
                <td>${escapeHtml(payment.first_name)}</td>
                <td>${formatNumber(payment.amount)} تومان</td>
                <td>${getPaymentMethodText(payment.payment_method)}</td>
                <td><span class="badge ${statusClass}">${statusText}</span></td>
                <td>${createdAt}</td>
                <td>
                    ${payment.status === 'pending' ? `
                        <div class="btn-group btn-group-sm">
                            <button class="btn btn-success" onclick="approvePayment(${payment.id})"
                                    title="تایید">
                                <i class="bi bi-check"></i>
                            </button>
                            <button class="btn btn-danger" onclick="rejectPayment(${payment.id})"
                                    title="رد">
                                <i class="bi bi-x"></i>
                            </button>
                        </div>
                    ` : `
                        <button class="btn btn-outline-primary btn-sm" onclick="viewPayment(${payment.id})"
                                title="مشاهده">
                            <i class="bi bi-eye"></i>
                        </button>
                    `}
                </td>
            </tr>
        `;
    });
    
    html += '</tbody></table></div>';
    container.innerHTML = html;
}

/**
 * Get payment status CSS class
 */
function getPaymentStatusClass(status) {
    const classes = {
        'pending': 'bg-warning',
        'completed': 'bg-success',
        'failed': 'bg-danger',
        'cancelled': 'bg-secondary',
        'reviewing': 'bg-info'
    };
    return classes[status] || 'bg-secondary';
}

/**
 * Get payment status text
 */
function getPaymentStatusText(status) {
    const texts = {
        'pending': 'در انتظار',
        'completed': 'تکمیل شده',
        'failed': 'ناموفق',
        'cancelled': 'لغو شده',
        'reviewing': 'در حال بررسی'
    };
    return texts[status] || 'نامشخص';
}

/**
 * Get payment method text
 */
function getPaymentMethodText(method) {
    const texts = {
        'zarinpal': 'زرین‌پال',
        'card_to_card': 'کارت به کارت',
        'crypto': 'ارز دیجیتال',
        'admin': 'ادمین'
    };
    return texts[method] || 'نامشخص';
}

/**
 * Approve payment
 */
function approvePayment(paymentId) {
    if (!confirm('آیا از تایید این پرداخت اطمینان دارید؟')) {
        return;
    }
    
    const formData = new FormData();
    formData.append('action', 'approve_payment');
    formData.append('payment_id', paymentId);
    
    fetch('ajax.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('پرداخت با موفقیت تایید شد', 'success');
            loadPaymentsData();
            loadDashboardData(); // Refresh stats
        } else {
            showNotification('خطا در تایید پرداخت: ' + (data.error || 'خطای نامشخص'), 'error');
        }
    })
    .catch(error => {
        console.error('Error approving payment:', error);
        showNotification('خطا در تایید پرداخت', 'error');
    });
}

/**
 * Reject payment
 */
function rejectPayment(paymentId) {
    const reason = prompt('دلیل رد پرداخت را وارد کنید (اختیاری):');
    if (reason === null) {
        return; // User cancelled
    }
    
    const formData = new FormData();
    formData.append('action', 'reject_payment');
    formData.append('payment_id', paymentId);
    formData.append('reason', reason);
    
    fetch('ajax.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showNotification('پرداخت رد شد', 'success');
            loadPaymentsData();
            loadDashboardData(); // Refresh stats
        } else {
            showNotification('خطا در رد پرداخت: ' + (data.error || 'خطای نامشخص'), 'error');
        }
    })
    .catch(error => {
        console.error('Error rejecting payment:', error);
        showNotification('خطا در رد پرداخت', 'error');
    });
}

/**
 * Utility Functions
 */

/**
 * Format number with thousands separator
 */
function formatNumber(num) {
    return new Intl.NumberFormat('fa-IR').format(num);
}

/**
 * Escape HTML to prevent XSS
 */
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

/**
 * Get time ago string
 */
function getTimeAgo(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const diffInSeconds = (now - date) / 1000;
    
    if (diffInSeconds < 60) {
        return 'همین الان';
    } else if (diffInSeconds < 3600) {
        const minutes = Math.floor(diffInSeconds / 60);
        return `${minutes} دقیقه پیش`;
    } else if (diffInSeconds < 86400) {
        const hours = Math.floor(diffInSeconds / 3600);
        return `${hours} ساعت پیش`;
    } else {
        const days = Math.floor(diffInSeconds / 86400);
        return `${days} روز پیش`;
    }
}

/**
 * Show notification
 */
function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `alert alert-${type === 'error' ? 'danger' : type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(notification);
    
    // Auto remove after 5 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 5000);
    
    // Play notification sound
    if (type === 'success' && notificationSound) {
        notificationSound.play().catch(() => {
            // Ignore audio play errors
        });
    }
}

/**
 * Start auto refresh for dashboard
 */
function startAutoRefresh() {
    refreshInterval = setInterval(() => {
        if (currentTab === 'dashboard') {
            loadDashboardData();
            loadRecentActivities();
        }
    }, 60000); // Refresh every minute
}

/**
 * Check for updates
 */
function checkForUpdates() {
    fetch('ajax.php?action=check_updates')
        .then(response => response.json())
        .then(data => {
            if (data.new_notifications) {
                showNotification(`${data.new_notifications} اعلان جدید دریافت شد`, 'info');
                
                // Update current tab if needed
                if (currentTab === 'payments' && data.new_payments) {
                    loadPaymentsData();
                }
            }
        })
        .catch(error => {
            console.error('Error checking updates:', error);
        });
}

/**
 * Load notification sound
 */
function loadNotificationSound() {
    try {
        notificationSound = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmYdAyKI0fPShykAJGZ2y9aOOgcWZLbr66hWEwdMoeT0wGMZhGb');
    } catch (error) {
        // Ignore audio loading errors
    }
}

/**
 * Export data functionality
 */
function exportData(type) {
    const url = `ajax.php?action=export_data&type=${type}`;
    window.open(url, '_blank');
}

/**
 * Search functionality
 */
function searchData(searchTerm, type) {
    const formData = new FormData();
    formData.append('action', 'search');
    formData.append('term', searchTerm);
    formData.append('type', type);
    
    fetch('ajax.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Update table with search results
            if (type === 'users') {
                updateUsersTable(data.results);
            } else if (type === 'payments') {
                updatePaymentsTable(data.results);
            }
        }
    })
    .catch(error => {
        console.error('Error searching:', error);
        showNotification('خطا در جستجو', 'error');
    });
}

// Global functions for inline event handlers
window.showAddMenuModal = showAddMenuModal;
window.editMenu = editMenu;
window.saveMenu = saveMenu;
window.deleteMenu = deleteMenu;
window.toggleMenu = toggleMenu;
window.approvePayment = approvePayment;
window.rejectPayment = rejectPayment;
window.exportData = exportData;
window.searchData = searchData;