<?php
/**
 * Admin Panel AJAX Handler
 * Captain Trade Bot - Professional AJAX API
 * Updated: 2025-10-15 16:42:33 UTC
 * Created by: saportinsta65
 */

session_start();

// Check admin authentication
if (!isset($_SESSION['admin_logged_in'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

require_once '../config.php';
require_once '../includes/database.php';
require_once '../includes/telegram.php';
require_once '../includes/payment.php';
require_once '../includes/spotplayer.php';
require_once '../includes/messages.php';

// Initialize components
$db = new Database();
$telegram = new TelegramBot(BOT_TOKEN);

// Get action
$action = $_GET['action'] ?? $_POST['action'] ?? '';

// Set JSON header
header('Content-Type: application/json');

try {
    switch ($action) {
        case 'get_stats':
            handleGetStats();
            break;
            
        case 'get_recent_activities':
            handleGetRecentActivities();
            break;
            
        case 'get_users':
            handleGetUsers();
            break;
            
        case 'get_menus':
            handleGetMenus();
            break;
            
        case 'get_menu':
            handleGetMenu();
            break;
            
        case 'add_menu':
            handleAddMenu();
            break;
            
        case 'update_menu':
            handleUpdateMenu();
            break;
            
        case 'delete_menu':
            handleDeleteMenu();
            break;
            
        case 'toggle_menu':
            handleToggleMenu();
            break;
            
        case 'get_payments':
            handleGetPayments();
            break;
            
        case 'approve_payment':
            handleApprovePayment();
            break;
            
        case 'reject_payment':
            handleRejectPayment();
            break;
            
        case 'toggle_user_block':
            handleToggleUserBlock();
            break;
            
        case 'search':
            handleSearch();
            break;
            
        case 'export_data':
            handleExportData();
            break;
            
        case 'check_updates':
            handleCheckUpdates();
            break;
            
        case 'get_analytics':
            handleGetAnalytics();
            break;
            
        case 'get_settings':
            handleGetSettings();
            break;
            
        case 'save_settings':
            handleSaveSettings();
            break;
            
        case 'test_spotplayer':
            handleTestSpotPlayer();
            break;
            
        case 'send_broadcast':
            handleSendBroadcast();
            break;
            
        default:
            throw new Exception('Invalid action');
    }
    
} catch (Exception $e) {
    error_log("AJAX Error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}

/**
 * Get dashboard statistics
 */
function handleGetStats() {
    global $db;
    
    $stats = $db->getAdminStats();
    
    echo json_encode([
        'success' => true,
        'stats' => $stats
    ]);
}

/**
 * Get recent activities
 */
function handleGetRecentActivities() {
    global $db;
    
    $activities = $db->getRecentActivities(20);
    
    echo json_encode([
        'success' => true,
        'activities' => $activities
    ]);
}

/**
 * Get users list
 */
function handleGetUsers() {
    global $db;
    
    $page = (int)($_GET['page'] ?? 1);
    $limit = (int)($_GET['limit'] ?? 50);
    $offset = ($page - 1) * $limit;
    
    $users = $db->getAllUsers($offset, $limit);
    
    echo json_encode([
        'success' => true,
        'users' => $users,
        'page' => $page
    ]);
}

/**
 * Get menus list
 */
function handleGetMenus() {
    global $db;
    
    $stmt = $db->connection->prepare("
        SELECT * FROM student_menus 
        ORDER BY sort_order ASC, created_at DESC
    ");
    $stmt->execute();
    $menus = $stmt->fetchAll();
    
    echo json_encode([
        'success' => true,
        'menus' => $menus
    ]);
}

/**
 * Get single menu
 */
function handleGetMenu() {
    global $db;
    
    $menu_id = (int)($_GET['id'] ?? 0);
    
    if (!$menu_id) {
        throw new Exception('Menu ID is required');
    }
    
    $stmt = $db->connection->prepare("SELECT * FROM student_menus WHERE id = ?");
    $stmt->execute([$menu_id]);
    $menu = $stmt->fetch();
    
    if (!$menu) {
        throw new Exception('Menu not found');
    }
    
    echo json_encode([
        'success' => true,
        'menu' => $menu
    ]);
}

/**
 * Add new menu
 */
function handleAddMenu() {
    global $db;
    
    $title = trim($_POST['title'] ?? '');
    $content = trim($_POST['content'] ?? '');
    $type = $_POST['type'] ?? 'text';
    $icon = $_POST['icon'] ?? '📋';
    
    if (empty($title) || empty($content)) {
        throw new Exception('Title and content are required');
    }
    
    // Validate content type
    $valid_types = ['text', 'link', 'file', 'video', 'channel', 'faq'];
    if (!in_array($type, $valid_types)) {
        $type = 'text';
    }
    
    $result = $db->addStudentMenu($title, $content, $type, $icon);
    
    if (!$result) {
        throw new Exception('Failed to add menu');
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Menu added successfully'
    ]);
}

/**
 * Update existing menu
 */
function handleUpdateMenu() {
    global $db;
    
    $menu_id = (int)($_POST['id'] ?? 0);
    $title = trim($_POST['title'] ?? '');
    $content = trim($_POST['content'] ?? '');
    $type = $_POST['type'] ?? 'text';
    $icon = $_POST['icon'] ?? '📋';
    
    if (!$menu_id || empty($title) || empty($content)) {
        throw new Exception('Menu ID, title and content are required');
    }
    
    $result = $db->updateStudentMenu($menu_id, $title, $content, $type, $icon);
    
    if (!$result) {
        throw new Exception('Failed to update menu');
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Menu updated successfully'
    ]);
}

/**
 * Delete menu
 */
function handleDeleteMenu() {
    global $db;
    
    $menu_id = (int)($_POST['id'] ?? 0);
    
    if (!$menu_id) {
        throw new Exception('Menu ID is required');
    }
    
    $result = $db->deleteStudentMenu($menu_id);
    
    if (!$result) {
        throw new Exception('Failed to delete menu');
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Menu deleted successfully'
    ]);
}

/**
 * Toggle menu active status
 */
function handleToggleMenu() {
    global $db;
    
    $menu_id = (int)($_POST['id'] ?? 0);
    $is_active = (bool)($_POST['is_active'] ?? false);
    
    if (!$menu_id) {
        throw new Exception('Menu ID is required');
    }
    
    $stmt = $db->connection->prepare("
        UPDATE student_menus SET is_active = ? WHERE id = ?
    ");
    $result = $stmt->execute([$is_active ? 1 : 0, $menu_id]);
    
    if (!$result) {
        throw new Exception('Failed to toggle menu status');
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Menu status updated successfully'
    ]);
}

/**
 * Get payments list
 */
function handleGetPayments() {
    global $db;
    
    $status = $_GET['status'] ?? 'all';
    $limit = (int)($_GET['limit'] ?? 50);
    
    $where_clause = '';
    $params = [];
    
    if ($status !== 'all') {
        $where_clause = 'WHERE p.status = ?';
        $params[] = $status;
    }
    
    $stmt = $db->connection->prepare("
        SELECT p.*, u.first_name, u.username, u.mobile_number
        FROM payments p
        JOIN users u ON p.user_id = u.user_id
        $where_clause
        ORDER BY p.created_at DESC
        LIMIT ?
    ");
    
    $params[] = $limit;
    $stmt->execute($params);
    $payments = $stmt->fetchAll();
    
    echo json_encode([
        'success' => true,
        'payments' => $payments
    ]);
}

/**
 * Approve payment
 */
function handleApprovePayment() {
    global $db;
    
    $payment_id = (int)($_POST['payment_id'] ?? 0);
    
    if (!$payment_id) {
        throw new Exception('Payment ID is required');
    }
    
    $result = CardPayment::approvePayment($payment_id, 'admin_panel');
    
    if (!$result['success']) {
        throw new Exception($result['error'] ?? 'Failed to approve payment');
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Payment approved successfully',
        'license_created' => $result['license_created'] ?? false
    ]);
}

/**
 * Reject payment
 */
function handleRejectPayment() {
    global $db;
    
    $payment_id = (int)($_POST['payment_id'] ?? 0);
    $reason = trim($_POST['reason'] ?? '');
    
    if (!$payment_id) {
        throw new Exception('Payment ID is required');
    }
    
    $result = CardPayment::rejectPayment($payment_id, $reason, 'admin_panel');
    
    if (!$result['success']) {
        throw new Exception($result['error'] ?? 'Failed to reject payment');
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Payment rejected successfully'
    ]);
}

/**
 * Toggle user block status
 */
function handleToggleUserBlock() {
    global $db;
    
    $user_id = (int)($_POST['user_id'] ?? 0);
    $blocked = (bool)($_POST['blocked'] ?? false);
    
    if (!$user_id) {
        throw new Exception('User ID is required');
    }
    
    $result = $db->toggleUserBlock($user_id, $blocked);
    
    if (!$result) {
        throw new Exception('Failed to update user status');
    }
    
    echo json_encode([
        'success' => true,
        'message' => $blocked ? 'User blocked successfully' : 'User unblocked successfully'
    ]);
}

/**
 * Search functionality
 */
function handleSearch() {
    global $db;
    
    $term = trim($_POST['term'] ?? '');
    $type = $_POST['type'] ?? 'users';
    
    if (empty($term)) {
        throw new Exception('Search term is required');
    }
    
    $results = [];
    
    switch ($type) {
        case 'users':
            $results = $db->searchUsers($term);
            break;
            
        case 'payments':
            $search = "%$term%";
            $stmt = $db->connection->prepare("
                SELECT p.*, u.first_name, u.username, u.mobile_number
                FROM payments p
                JOIN users u ON p.user_id = u.user_id
                WHERE u.first_name LIKE ? 
                   OR u.username LIKE ? 
                   OR u.mobile_number LIKE ?
                   OR p.id LIKE ?
                ORDER BY p.created_at DESC
                LIMIT 20
            ");
            $stmt->execute([$search, $search, $search, $search]);
            $results = $stmt->fetchAll();
            break;
            
        default:
            throw new Exception('Invalid search type');
    }
    
    echo json_encode([
        'success' => true,
        'results' => $results,
        'count' => count($results)
    ]);
}

/**
 * Export data
 */
function handleExportData() {
    global $db;
    
    $type = $_GET['type'] ?? 'users';
    $format = $_GET['format'] ?? 'csv';
    
    switch ($type) {
        case 'users':
            exportUsers($format);
            break;
            
        case 'payments':
            exportPayments($format);
            break;
            
        case 'activities':
            exportActivities($format);
            break;
            
        default:
            throw new Exception('Invalid export type');
    }
}

/**
 * Export users data
 */
function exportUsers($format) {
    global $db;
    
    $stmt = $db->connection->query("
        SELECT user_id, username, first_name, mobile_number, 
               is_verified, created_at, last_activity
        FROM users 
        ORDER BY created_at DESC
    ");
    $users = $stmt->fetchAll();
    
    if ($format === 'csv') {
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="users_export_' . date('Y-m-d') . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // Add BOM for UTF-8
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
        
        // CSV headers
        fputcsv($output, ['User ID', 'Username', 'First Name', 'Mobile', 'Verified', 'Created', 'Last Activity']);
        
        foreach ($users as $user) {
            fputcsv($output, [
                $user['user_id'],
                $user['username'],
                $user['first_name'],
                $user['mobile_number'],
                $user['is_verified'] ? 'Yes' : 'No',
                $user['created_at'],
                $user['last_activity']
            ]);
        }
        
        fclose($output);
    } else {
        // JSON format
        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="users_export_' . date('Y-m-d') . '.json"');
        
        echo json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    }
    
    exit;
}

/**
 * Export payments data
 */
function exportPayments($format) {
    global $db;
    
    $stmt = $db->connection->query("
        SELECT p.*, u.first_name, u.username, u.mobile_number
        FROM payments p
        JOIN users u ON p.user_id = u.user_id
        ORDER BY p.created_at DESC
    ");
    $payments = $stmt->fetchAll();
    
    if ($format === 'csv') {
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="payments_export_' . date('Y-m-d') . '.csv"');
        
        $output = fopen('php://output', 'w');
        
        // Add BOM for UTF-8
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));
        
        // CSV headers
        fputcsv($output, ['Payment ID', 'User', 'Amount', 'Method', 'Status', 'Created', 'Ref ID']);
        
        foreach ($payments as $payment) {
            fputcsv($output, [
                $payment['id'],
                $payment['first_name'],
                $payment['amount'],
                $payment['payment_method'],
                $payment['status'],
                $payment['created_at'],
                $payment['ref_id']
            ]);
        }
        
        fclose($output);
    } else {
        // JSON format
        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="payments_export_' . date('Y-m-d') . '.json"');
        
        echo json_encode($payments, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    }
    
    exit;
}

/**
 * Check for updates
 */
function handleCheckUpdates() {
    global $db;
    
    // Check for new notifications in last minute
    $stmt = $db->connection->query("
        SELECT COUNT(*) as count 
        FROM user_activities 
        WHERE created_at > DATE_SUB(NOW(), INTERVAL 1 MINUTE)
    ");
    $new_notifications = $stmt->fetch()['count'];
    
    // Check for new payments
    $stmt = $db->connection->query("
        SELECT COUNT(*) as count 
        FROM payments 
        WHERE status = 'pending' 
        AND created_at > DATE_SUB(NOW(), INTERVAL 5 MINUTE)
    ");
    $new_payments = $stmt->fetch()['count'];
    
    echo json_encode([
        'success' => true,
        'new_notifications' => $new_notifications,
        'new_payments' => $new_payments
    ]);
}

/**
 * Get analytics data
 */
function handleGetAnalytics() {
    global $db;
    
    // Get daily registrations for last 30 days
    $stmt = $db->connection->query("
        SELECT DATE(created_at) as date, COUNT(*) as count
        FROM users
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY DATE(created_at)
        ORDER BY date ASC
    ");
    $daily_registrations = $stmt->fetchAll();
    
    // Get payment methods stats
    $stmt = $db->connection->query("
        SELECT payment_method, COUNT(*) as count, SUM(amount) as total
        FROM payments
        WHERE status = 'completed'
        GROUP BY payment_method
    ");
    $payment_methods = $stmt->fetchAll();
    
    // Get popular menus
    $stmt = $db->connection->query("
        SELECT title, access_count
        FROM student_menus
        WHERE is_active = 1
        ORDER BY access_count DESC
        LIMIT 10
    ");
    $popular_menus = $stmt->fetchAll();
    
    echo json_encode([
        'success' => true,
        'analytics' => [
            'daily_registrations' => $daily_registrations,
            'payment_methods' => $payment_methods,
            'popular_menus' => $popular_menus
        ]
    ]);
}

/**
 * Get settings
 */
function handleGetSettings() {
    global $db;
    
    $stmt = $db->connection->query("
        SELECT setting_key, setting_value, setting_type, description, category
        FROM admin_settings
        ORDER BY category, setting_key
    ");
    $settings = $stmt->fetchAll();
    
    echo json_encode([
        'success' => true,
        'settings' => $settings
    ]);
}

/**
 * Save settings
 */
function handleSaveSettings() {
    global $db;
    
    $settings = $_POST['settings'] ?? [];
    
    if (empty($settings)) {
        throw new Exception('No settings provided');
    }
    
    foreach ($settings as $key => $value) {
        $type = $_POST['types'][$key] ?? 'string';
        $db->setAdminSetting($key, $value, $type, 'admin_panel');
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Settings saved successfully'
    ]);
}

/**
 * Test SpotPlayer connection
 */
function handleTestSpotPlayer() {
    $spotPlayer = new SpotPlayerAPI();
    $result = $spotPlayer->testConnection();
    
    echo json_encode($result);
}

/**
 * Send broadcast message
 */
function handleSendBroadcast() {
    global $db, $telegram;
    
    $message = trim($_POST['message'] ?? '');
    $target = $_POST['target'] ?? 'all';
    
    if (empty($message)) {
        throw new Exception('Message is required');
    }
    
    // Get target users
    $where_clause = '';
    if ($target === 'verified') {
        $where_clause = 'WHERE is_verified = 1';
    } elseif ($target === 'unverified') {
        $where_clause = 'WHERE is_verified = 0';
    }
    
    $stmt = $db->connection->prepare("
        SELECT user_id, chat_id, first_name 
        FROM users 
        $where_clause
        AND is_blocked = 0
    ");
    $stmt->execute();
    $users = $stmt->fetchAll();
    
    $sent_count = 0;
    $failed_count = 0;
    
    foreach ($users as $user) {
        try {
            $result = $telegram->sendMessage($user['chat_id'], $message);
            if ($result) {
                $sent_count++;
            } else {
                $failed_count++;
            }
            
            // Small delay to avoid rate limiting
            usleep(100000); // 0.1 second
            
        } catch (Exception $e) {
            $failed_count++;
            error_log("Broadcast failed for user {$user['user_id']}: " . $e->getMessage());
        }
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Broadcast completed',
        'sent' => $sent_count,
        'failed' => $failed_count,
        'total' => count($users)
    ]);
}
?>