<?php
/**
 * File Upload & Processing System
 * Updated: 2025-10-15 19:13:08 UTC
 * Created by: saportinsta65
 */

class FileHandler {
    private $telegram;
    private $db;
    
    public function __construct($telegram, $db) {
        $this->telegram = $telegram;
        $this->db = $db;
    }
    
    /**
     * 📸 پردازش آپلود عکس
     */
    public function handlePhotoUpload($chat_id, $user_id, $photo) {
        $user_state = $this->db->getUserState($user_id);
        
        if ($user_state === 'awaiting_receipt') {
            return $this->processReceiptPhoto($chat_id, $user_id, $photo);
        }
        
        $this->telegram->sendMessage(
            $chat_id, 
            "🤔 در حال حاضر انتظار دریافت عکس نیستم.\n\n💡 از منو استفاده کنید."
        );
        
        return false;
    }
    
    /**
     * 💳 پردازش عکس رسید
     */
    private function processReceiptPhoto($chat_id, $user_id, $photo) {
        try {
            // Get largest photo
            $largest_photo = end($photo);
            $file_id = $largest_photo['file_id'];
            
            // Get file info from Telegram
            $file_info = $this->telegram->getFile($file_id);
            
            if (!$file_info || !isset($file_info['result']['file_path'])) {
                throw new Exception("Failed to get file info");
            }
            
            $file_path = $file_info['result']['file_path'];
            $file_url = "https://api.telegram.org/file/bot" . BOT_TOKEN . "/" . $file_path;
            
            // Get user payment info
            $user_info = $this->db->getUserInfo($user_id);
            $payment_id = $user_info['payment_id'];
            
            if (!$payment_id) {
                throw new Exception("No payment ID found");
            }
            
            // Save receipt info
            $this->saveReceiptInfo($payment_id, $file_id, $file_url, $largest_photo);
            
            // Update payment status
            $this->db->updatePaymentReceipt($payment_id, $file_url);
            $this->db->setUserState($user_id, 'payment_pending');
            
            // Send confirmation
            $this->sendReceiptConfirmation($chat_id, $payment_id);
            
            // Notify admin
            $this->notifyAdminNewReceipt($payment_id, $user_info, $file_url);
            
            return true;
            
        } catch (Exception $e) {
            error_log("Receipt processing error: " . $e->getMessage());
            $this->telegram->sendMessage($chat_id, "❌ خطا در پردازش رسید. لطفاً دوباره تلاش کنید.");
            return false;
        }
    }
    
    /**
     * 💾 ذخیره اطلاعات رسید
     */
    private function saveReceiptInfo($payment_id, $file_id, $file_url, $photo_info) {
        try {
            $stmt = $this->db->connection->prepare("
                INSERT INTO payment_receipts 
                (payment_id, file_id, file_url, file_type, file_size, uploaded_at) 
                VALUES (?, ?, ?, 'photo', ?, NOW())
            ");
            
            $file_size = $photo_info['file_size'] ?? 0;
            $stmt->execute([$payment_id, $file_id, $file_url, $file_size]);
            
        } catch (Exception $e) {
            error_log("Save receipt info error: " . $e->getMessage());
        }
    }
    
    /**
     * ✅ ارسال تایید دریافت رسید
     */
    private function sendReceiptConfirmation($chat_id, $payment_id) {
        $message = "✅ <b>رسید با موفقیت دریافت شد!</b>

━━━━━━━━━━━━━━━━━━━━

🆔 <b>کد پیگیری:</b> <code>{$payment_id}</code>

⏳ <b>وضعیت:</b> در صف بررسی

━━━━━━━━━━━━━━━━━━━━

⏰ <b>زمان بررسی:</b> 15-30 دقیقه
📬 <b>پس از تایید لایسنس ارسال می‌شود</b>

━━━━━━━━━━━━━━━━━━━━

📞 <b>پیگیری:</b> " . SUPPORT_USERNAME . "

💎 <b>ممنون از اعتماد شما!</b>";
        
        $keyboard = [
            'keyboard' => [
                [
                    ['text' => '🏠 منوی اصلی'],
                    ['text' => '💬 پشتیبانی']
                ]
            ],
            'resize_keyboard' => true,
            'one_time_keyboard' => false
        ];
        
        $this->telegram->sendMessage($chat_id, $message, $keyboard);
    }
    
    /**
     * 🔔 اطلاع‌رسانی به ادمین
     */
    private function notifyAdminNewReceipt($payment_id, $user_info, $receipt_url) {
        if (!defined('ADMIN_CHAT_ID') || empty(ADMIN_CHAT_ID)) {
            return;
        }
        
        $message = "🔔 <b>رسید جدید دریافت شد!</b>

━━━━━━━━━━━━━━━━━━━━

🆔 <b>کد پرداخت:</b> <code>{$payment_id}</code>

👤 <b>کاربر:</b> {$user_info['first_name']}
📱 <b>یوزرنیم:</b> @{$user_info['username']}
🔢 <b>آیدی:</b> <code>{$user_info['user_id']}</code>

💰 <b>مبلغ:</b> " . number_format(COURSE_PRICE) . " تومان

━━━━━━━━━━━━━━━━━━━━

🎯 <b>اقدامات سریع:</b>
✅ <code>/ok {$payment_id}</code> - تایید
❌ <code>/no {$payment_id}</code> - رد
🔍 <code>/show {$payment_id}</code> - جزئیات

📸 <b>رسید در پیام بعدی...</b>";
        
        $admin_keyboard = [
            'keyboard' => [
                [
                    ['text' => '💳 بررسی پرداخت‌ها'],
                    ['text' => '👨‍💼 پنل مدیریت']
                ]
            ],
            'resize_keyboard' => true,
            'one_time_keyboard' => false
        ];
        
        $this->telegram->sendMessage(ADMIN_CHAT_ID, $message, $admin_keyboard);
        
        // Send receipt photo
        $this->telegram->sendPhoto(
            ADMIN_CHAT_ID, 
            $receipt_url, 
            "📸 رسید پرداخت {$payment_id}"
        );
    }
    
    /**
     * 📄 پردازش آپلود سند
     */
    public function handleDocumentUpload($chat_id, $user_id, $document) {
        $user_state = $this->db->getUserState($user_id);
        
        if ($user_state === 'awaiting_receipt') {
            return $this->processReceiptDocument($chat_id, $user_id, $document);
        }
        
        $this->telegram->sendMessage(
            $chat_id, 
            "📄 فایل دریافت شد ولی در حال حاضر انتظار فایلی نیستم.\n\n💡 از منو استفاده کنید."
        );
        
        return false;
    }
    
    /**
     * 📄 پردازش سند رسید
     */
    private function processReceiptDocument($chat_id, $user_id, $document) {
        try {
            $file_id = $document['file_id'];
            $file_name = $document['file_name'] ?? 'receipt_document';
            $file_size = $document['file_size'] ?? 0;
            
            // Check file type
            $allowed_types = ['pdf', 'jpg', 'jpeg', 'png'];
            $extension = pathinfo($file_name, PATHINFO_EXTENSION);
            
            if (!in_array(strtolower($extension), $allowed_types)) {
                $this->telegram->sendMessage(
                    $chat_id, 
                    "❌ فرمت فایل مجاز نیست.\n\n✅ فرمت‌های مجاز: PDF, JPG, PNG"
                );
                return false;
            }
            
            // Get file URL
            $file_info = $this->telegram->getFile($file_id);
            
            if (!$file_info || !isset($file_info['result']['file_path'])) {
                throw new Exception("Failed to get document info");
            }
            
            $file_path = $file_info['result']['file_path'];
            $file_url = "https://api.telegram.org/file/bot" . BOT_TOKEN . "/" . $file_path;
            
            // Save and process like photo receipt
            $user_info = $this->db->getUserInfo($user_id);
            $payment_id = $user_info['payment_id'];
            
            if (!$payment_id) {
                throw new Exception("No payment ID found");
            }
            
            // Save document info
            $this->saveDocumentInfo($payment_id, $file_id, $file_url, $document);
            
            // Update payment
            $this->db->updatePaymentReceipt($payment_id, $file_url);
            $this->db->setUserState($user_id, 'payment_pending');
            
            // Send confirmation
            $this->sendReceiptConfirmation($chat_id, $payment_id);
            
            // Notify admin
            $this->notifyAdminNewDocument($payment_id, $user_info, $file_url, $file_name);
            
            return true;
            
        } catch (Exception $e) {
            error_log("Document processing error: " . $e->getMessage());
            $this->telegram->sendMessage($chat_id, "❌ خطا در پردازش فایل. لطفاً دوباره تلاش کنید.");
            return false;
        }
    }
    
    /**
     * 💾 ذخیره اطلاعات سند
     */
    private function saveDocumentInfo($payment_id, $file_id, $file_url, $document) {
        try {
            $stmt = $this->db->connection->prepare("
                INSERT INTO payment_receipts 
                (payment_id, file_id, file_url, file_type, file_size, uploaded_at) 
                VALUES (?, ?, ?, 'document', ?, NOW())
            ");
            
            $file_size = $document['file_size'] ?? 0;
            $stmt->execute([$payment_id, $file_id, $file_url, $file_size]);
            
        } catch (Exception $e) {
            error_log("Save document info error: " . $e->getMessage());
        }
    }
    
    /**
     * 🔔 اطلاع‌رسانی سند به ادمین
     */
    private function notifyAdminNewDocument($payment_id, $user_info, $file_url, $file_name) {
        if (!defined('ADMIN_CHAT_ID') || empty(ADMIN_CHAT_ID)) {
            return;
        }
        
        $message = "📄 <b>سند رسید جدید!</b>

🆔 کد: <code>{$payment_id}</code>
👤 {$user_info['first_name']} (@{$user_info['username']})
📄 فایل: {$file_name}

🎯 <code>/ok {$payment_id}</code> - تایید
❌ <code>/no {$payment_id}</code> - رد";
        
        $this->telegram->sendMessage(ADMIN_CHAT_ID, $message);
        $this->telegram->sendDocument(ADMIN_CHAT_ID, $file_url, "📄 سند پرداخت {$payment_id}");
    }
    
    /**
     * 🗑️ حذف فایل‌های قدیمی
     */
    public function cleanupOldFiles($days = 30) {
        try {
            $stmt = $this->db->connection->prepare("
                DELETE FROM payment_receipts 
                WHERE uploaded_at < DATE_SUB(NOW(), INTERVAL ? DAY)
            ");
            
            $stmt->execute([$days]);
            $deleted = $stmt->rowCount();
            
            error_log("Cleaned up {$deleted} old receipt files");
            return $deleted;
            
        } catch (Exception $e) {
            error_log("File cleanup error: " . $e->getMessage());
            return 0;
        }
    }
}
?>