<?php
/**
 * مدیریت تمرین‌ها و یکپارچگی با سیستم کمپین
 * نسخه تصحیح شده - 12 اکتبر 2025
 * رفع مشکلات circular dependency و تداخل توابع
 */

// محافظت از دسترسی مستقیم
if (!defined('BOT_ACCESS')) {
    die('Access Denied');
}

// ✅ حذف require_once چون از main.php فراخوانی می‌شه
// require_once 'functions.php';
// require_once 'config.php';

// بارگذاری امن فایل‌های وابسته
if (file_exists('campaign.php') && !function_exists('startCampaign')) {
    require_once 'campaign.php';
}

/**
 * ✅ تابع کمکی برای پردازش ایمن JSON - بهبود یافته
 */
if (!function_exists('safeJsonDecode')) {
    function safeJsonDecode($data, $default = []) {
        if (empty($data)) {
            return $default;
        }
        
        if (is_array($data)) {
            return $data;
        }
        
        if (is_string($data)) {
            $decoded = json_decode($data, true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                return $decoded;
            }
            // لاگ خطای JSON برای دیباگ
            error_log("JSON decode error for data: " . substr($data, 0, 100) . " - Error: " . json_last_error_msg());
        }
        
        return $default;
    }
}

/**
 * ✅ تابع کمکی برای debug logging - فقط اگر وجود نداشته باشه
 */
if (!function_exists('exerciseDebugLog')) {
    function exerciseDebugLog($message, $data = null) {
        $log_message = "[EXERCISE_DEBUG] $message";
        if ($data !== null) {
            $log_message .= " - Data: " . json_encode($data, JSON_UNESCAPED_UNICODE);
        }
        error_log($log_message);
    }
}

/**
 * ارسال تمرین جلسه به کاربر - بهبود یافته
 */
function sendExercise($user_id, $session_title) {
    try {
        exerciseDebugLog("Sending exercise", ['user_id' => $user_id, 'session' => $session_title]);
        
        if (!function_exists('loadSessions')) {
            error_log("loadSessions function not found");
            return false;
        }
        
        $sessions = loadSessions();
        $session = null;
        
        foreach ($sessions as $sess) {
            if ($sess['title'] == $session_title) {
                $session = $sess;
                break;
            }
        }
        
        if (!$session || !isset($session['exercise']) || !$session['exercise']) {
            error_log("No exercise found for session: $session_title");
            return false;
        }
        
        $exercise_text = $session['exercise'];
        $message = "📝 <b>تمرین این جلسه:</b>\n\n" . $exercise_text . "\n\nلطفاً پاسخ تمرین را به صورت پیام متنی ارسال کنید.";
        
        if (!function_exists('sendMessage')) {
            error_log("sendMessage function not found");
            return false;
        }
        
        $result = sendMessage($user_id, $message);
        
        if ($result) {
            // وضعیت تمرین را pending بگذاریم
            if (!function_exists('getUserById')) {
                error_log("getUserById function not found");
                return false;
            }
            
            $user = getUserById($user_id);
            $exercises = safeJsonDecode($user['exercises'] ?? null, []);
            
            // ✅ اطمینان از integer بودن session ID
            $session_id = intval($session['id']);
            
            if (!isset($exercises[$session_id])) {
                $exercises[$session_id] = [
                    'answer' => '',
                    'status' => 'pending',
                    'sent_at' => date('Y-m-d H:i:s'),
                    'session_title' => $session_title
                ];
                $user['exercises'] = json_encode($exercises, JSON_UNESCAPED_UNICODE);
                
                if (function_exists('saveUser') && saveUser($user)) {
                    exerciseDebugLog("Exercise pending status saved", ['user_id' => $user_id, 'session_id' => $session_id]);
                    error_log("Exercise sent to user $user_id for session: $session_title");
                } else {
                    error_log("Failed to save exercise pending status for user $user_id");
                }
            }
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log("Error sending exercise to user $user_id: " . $e->getMessage());
        exerciseDebugLog("Exception in sendExercise", ['error' => $e->getMessage()]);
        return false;
    }
}

/**
 * پردازش پاسخ تمرین کاربر - بهبود یافته
 */
function handleExerciseAnswer($user_id, $session_title, $answer) {
    try {
        exerciseDebugLog("Handling exercise answer", [
            'user_id' => $user_id, 
            'session' => $session_title, 
            'answer_length' => strlen($answer)
        ]);
        
        if (!function_exists('loadSessions') || !function_exists('getUserById') || !function_exists('saveUser')) {
            error_log("Required functions not found for handleExerciseAnswer");
            return false;
        }
        
        $sessions = loadSessions();
        $session = null;
        
        foreach ($sessions as $sess) {
            if ($sess['title'] == $session_title) {
                $session = $sess;
                break;
            }
        }
        
        if (!$session) {
            error_log("Session not found: $session_title");
            return false;
        }
        
        // ✅ اطمینان از integer بودن session ID
        $session_id = intval($session['id']);
        
        // ذخیره پاسخ کاربر
        $user = getUserById($user_id);
        $exercises = safeJsonDecode($user['exercises'] ?? null, []);
        
        $exercises[$session_id] = [
            'answer' => $answer,
            'status' => 'pending',
            'submitted_at' => date('Y-m-d H:i:s'),
            'session_title' => $session_title,
            'session_id' => $session_id
        ];
        
        $user['exercises'] = json_encode($exercises, JSON_UNESCAPED_UNICODE);
        
        if (!saveUser($user)) {
            error_log("Failed to save exercise answer for user $user_id");
            return false;
        }
        
        exerciseDebugLog("Exercise answer saved", ['user_id' => $user_id, 'session_id' => $session_id]);
        
        // اطلاع به کاربر
        if (function_exists('sendMessage')) {
            sendMessage($user_id, "✅ تمرین شما دریافت شد و در انتظار بررسی ادمین است.\n\nبه محض تایید، جلسه بعدی برای شما فعال خواهد شد.");
        }
        
        // ارسال برای ادمین با دکمه‌های inline
        if (defined('ADMIN_ID') && function_exists('sendMessage')) {
            $user_info = getUserById($user_id);
            $user_name = $user_info['first_name'] ?? 'کاربر';
            
            $admin_message = "📝 <b>تمرین جدید دریافت شد</b>\n\n";
            $admin_message .= "👤 کاربر: <b>$user_name</b> (<code>$user_id</code>)\n";
            $admin_message .= "📚 جلسه: <b>$session_title</b>\n";
            $admin_message .= "🆔 Session ID: <code>$session_id</code>\n";
            $admin_message .= "📝 پاسخ:\n\n<code>" . htmlspecialchars($answer) . "</code>";
            
            // ارسال پیام اول (اطلاعات)
            sendMessage(ADMIN_ID, $admin_message);
            
            // ارسال پیام دوم (دکمه‌ها)
            $reply_markup = [
                'inline_keyboard' => [
                    [
                        [
                            'text' => '✅ تایید تمرین',
                            'callback_data' => "exercise_accept_{$user_id}_{$session_id}"
                        ],
                        [
                            'text' => '❌ رد تمرین', 
                            'callback_data' => "exercise_reject_{$user_id}_{$session_id}"
                        ]
                    ],
                    [
                        [
                            'text' => '🔍 مشاهده کامل',
                            'callback_data' => "exercise_view_{$user_id}_{$session_id}"
                        ]
                    ]
                ]
            ];
            
            // ارسال دکمه‌ها با API مستقیم
            if (defined('API_URL')) {
                $url = API_URL . "sendMessage";
                $data = [
                    'chat_id' => ADMIN_ID,
                    'text' => "🔹 لطفاً تمرین را بررسی و تایید یا رد کنید:",
                    'reply_markup' => json_encode($reply_markup)
                ];
                
                $ch = curl_init($url);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                $response = curl_exec($ch);
                $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                curl_close($ch);
                
                if ($http_code !== 200) {
                    error_log("Failed to send exercise buttons to admin - HTTP: $http_code");
                }
            }
        }
        
        exerciseDebugLog("Exercise submitted successfully", [
            'user_id' => $user_id, 
            'session_id' => $session_id
        ]);
        
        error_log("Exercise answer submitted by user $user_id for session: $session_title");
        return true;
        
    } catch (Exception $e) {
        error_log("Error handling exercise answer for user $user_id: " . $e->getMessage());
        exerciseDebugLog("Exception in handleExerciseAnswer", ['error' => $e->getMessage()]);
        return false;
    }
}

/**
 * ✅ پردازش callback های تایید/رد/مشاهده تمرین - نسخه کامل و امن
 */
function handleExerciseCallback($data) {
    try {
        exerciseDebugLog("Processing exercise callback", ['data' => $data]);
        
        // بررسی توابع مورد نیاز
        if (!function_exists('getUserById') || !function_exists('saveUser') || !function_exists('sendMessage')) {
            error_log("Required functions not found for handleExerciseCallback");
            return false;
        }
        
        // ✅ مشاهده کامل تمرین
        if (preg_match('/^exercise_view_([0-9]+)_([0-9]+)$/', $data, $matches)) {
            $user_id = intval($matches[1]);
            $session_id = intval($matches[2]);
            
            exerciseDebugLog("Exercise view callback", ['user_id' => $user_id, 'session_id' => $session_id]);
            
            $user = getUserById($user_id);
            if (!$user) {
                exerciseDebugLog("User not found for exercise view", ['user_id' => $user_id]);
                if (defined('ADMIN_ID')) {
                    sendMessage(ADMIN_ID, "❌ کاربر یافت نشد: $user_id");
                }
                return false;
            }
            
            $exercises = safeJsonDecode($user['exercises'] ?? null, []);
            
            // ✅ بررسی هر دو حالت: string و integer session_id
            $exercise = null;
            if (isset($exercises[$session_id])) {
                $exercise = $exercises[$session_id];
            } elseif (isset($exercises[strval($session_id)])) {
                $exercise = $exercises[strval($session_id)];
            }
            
            if (!$exercise) {
                exerciseDebugLog("Exercise not found for user", [
                    'user_id' => $user_id, 
                    'session_id' => $session_id,
                    'available_sessions' => array_keys($exercises)
                ]);
                if (defined('ADMIN_ID')) {
                    sendMessage(ADMIN_ID, "❌ تمرین برای این کاربر و جلسه یافت نشد.\n\nکاربر: $user_id\nجلسه: $session_id\nموجود: " . implode(', ', array_keys($exercises)));
                }
                return false;
            }
            
            // پیدا کردن نام جلسه
            $session_title = $exercise['session_title'] ?? '';
            if (empty($session_title)) {
                if (function_exists('loadSessions')) {
                    $sessions = loadSessions();
                    foreach ($sessions as $sess) {
                        if (intval($sess['id']) == intval($session_id)) {
                            $session_title = $sess['title'];
                            break;
                        }
                    }
                }
            }
            
            if (empty($session_title)) {
                $session_title = "جلسه شماره $session_id";
            }
            
            $detailed_msg = "🔍 <b>جزئیات کامل تمرین</b>\n\n";
            $detailed_msg .= "👤 کاربر: <b>{$user['first_name']}</b> (#{$user_id})\n";
            $detailed_msg .= "📚 جلسه: <b>$session_title</b>\n";
            $detailed_msg .= "🆔 Session ID: <code>$session_id</code>\n";
            $detailed_msg .= "📅 زمان ارسال: " . ($exercise['submitted_at'] ?? 'نامشخص') . "\n";
            $detailed_msg .= "📊 وضعیت: <b>منتظر بررسی</b>\n\n";
            $detailed_msg .= "💬 <b>متن کامل تمرین:</b>\n";
            $detailed_msg .= "─────────────────────────\n";
            $detailed_msg .= ($exercise['answer'] ?? 'پاسخ خالی');
            $detailed_msg .= "\n─────────────────────────\n\n";
            
            // اطلاعات اضافی کاربر
            if (function_exists('loadSessions')) {
                $seen_sessions = safeJsonDecode($user['seen_sessions'] ?? null, []);
                $seen_count = is_array($seen_sessions) ? count($seen_sessions) : 0;
                
                $sessions = loadSessions();
                $total_sessions = count($sessions);
                
                $detailed_msg .= "📈 <b>پیشرفت کاربر:</b>\n";
                $detailed_msg .= "جلسات دیده شده: $seen_count / $total_sessions\n";
                if ($total_sessions > 0) {
                    $detailed_msg .= "درصد پیشرفت: " . round(($seen_count / $total_sessions) * 100) . "%\n\n";
                }
            }
            
            $detailed_msg .= "🕒 زمان نمایش: " . date('H:i:s');
            
            // ارسال پیام جزئیات
            if (defined('ADMIN_ID')) {
                sendMessage(ADMIN_ID, $detailed_msg);
                
                // ارسال دکمه‌های تایید/رد بعد از نمایش جزئیات
                $reply_markup = [
                    'inline_keyboard' => [
                        [
                            [
                                'text' => '✅ تایید تمرین',
                                'callback_data' => "exercise_accept_{$user_id}_{$session_id}"
                            ],
                            [
                                'text' => '❌ رد تمرین',
                                'callback_data' => "exercise_reject_{$user_id}_{$session_id}"
                            ]
                        ]
                    ]
                ];
                
                if (defined('API_URL')) {
                    $url = API_URL . "sendMessage";
                    $data_send = [
                        'chat_id' => ADMIN_ID,
                        'text' => "🎯 <b>اکنون می‌توانید تمرین را تایید یا رد کنید:</b>",
                        'reply_markup' => json_encode($reply_markup)
                    ];
                    
                    $ch = curl_init($url);
                    curl_setopt($ch, CURLOPT_POST, true);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $data_send);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                    curl_exec($ch);
                    curl_close($ch);
                }
            }
            
            exerciseDebugLog("Exercise view completed", ['user_id' => $user_id, 'session_id' => $session_id]);
            return true;
        }
        
        // تایید تمرین
        if (preg_match('/^exercise_accept_([0-9]+)_([0-9]+)$/', $data, $matches)) {
            $user_id = intval($matches[1]);
            $session_id = $matches[2]; // نگه داشتن type اصلی
            
            exerciseDebugLog("Exercise accept callback", ['user_id' => $user_id, 'session_id' => $session_id]);
            
            $user = getUserById($user_id);
            if (!$user) {
                error_log("User not found: $user_id");
                if (defined('ADMIN_ID')) {
                    sendMessage(ADMIN_ID, "❌ کاربر یافت نشد: $user_id");
                }
                return false;
            }
            
            $exercises = safeJsonDecode($user['exercises'] ?? null, []);
            exerciseDebugLog("User exercises data", ['exercises_keys' => array_keys($exercises)]);
            
            // ✅ بررسی هر دو حالت: string و integer session_id
            $exercise_key = null;
            if (isset($exercises[$session_id])) {
                $exercise_key = $session_id;
            } elseif (isset($exercises[intval($session_id)])) {
                $exercise_key = intval($session_id);
            } elseif (isset($exercises[strval($session_id)])) {
                $exercise_key = strval($session_id);
            }
            
            if ($exercise_key !== null) {
                $exercises[$exercise_key]['status'] = 'accepted';
                $exercises[$exercise_key]['approved_at'] = date('Y-m-d H:i:s');
                $user['exercises'] = json_encode($exercises, JSON_UNESCAPED_UNICODE);
                
                if (!saveUser($user)) {
                    error_log("Failed to save user after exercise accept");
                    if (defined('ADMIN_ID')) {
                        sendMessage(ADMIN_ID, "❌ خطا در ذخیره تایید تمرین");
                    }
                    return false;
                }
                
                // پیدا کردن نام جلسه
                $session_title = $exercises[$exercise_key]['session_title'] ?? "";
                
                if (empty($session_title)) {
                    if (function_exists('loadSessions')) {
                        $sessions = loadSessions();
                        foreach ($sessions as $sess) {
                            if (intval($sess['id']) == intval($session_id)) {
                                $session_title = $sess['title'];
                                break;
                            }
                        }
                    }
                }
                
                if (empty($session_title)) {
                    $session_title = "جلسه شماره $session_id";
                }
                
                // اطلاع به کاربر
                sendMessage($user_id, "🎉 <b>تبریک!</b>\n\nتمرین شما برای جلسه <b>$session_title</b> تایید شد!\n\nجلسه بعدی برای شما فعال شد.");
                
                // اطلاع به ادمین
                if (defined('ADMIN_ID')) {
                    sendMessage(ADMIN_ID, "✅ تمرین کاربر <code>$user_id</code> برای جلسه <b>$session_title</b> تایید شد.");
                }
                
                // بررسی اینکه آیا این آخرین جلسه است
                if (function_exists('isLastSession')) {
                    $is_last_session = isLastSession(intval($session_id));
                    
                    if ($is_last_session) {
                        // بررسی اینکه آیا کاربر واجد شرایط کمپین است
                        if (function_exists('isUserEligibleForCampaign') && isUserEligibleForCampaign($user_id)) {
                            error_log("User $user_id completed the course, starting campaign");
                            
                            // شروع کمپین ایمن
                            if (function_exists('startCampaign')) {
                                startCampaign($user_id);
                                if (defined('ADMIN_ID')) {
                                    sendMessage(ADMIN_ID, "🎯 کاربر <code>$user_id</code> دوره را تکمیل کرد و کمپین برای او شروع شد.");
                                }
                            } else {
                                error_log("startCampaign function not found");
                                if (defined('ADMIN_ID')) {
                                    sendMessage(ADMIN_ID, "⚠️ کاربر <code>$user_id</code> دوره را تکمیل کرد اما تابع کمپین یافت نشد.");
                                }
                            }
                        } else {
                            error_log("User $user_id completed course but not eligible for campaign");
                            if (defined('ADMIN_ID')) {
                                sendMessage(ADMIN_ID, "⚠️ کاربر <code>$user_id</code> دوره را تکمیل کرد اما واجد شرایط کمپین نیست.");
                            }
                        }
                    }
                }
                
                exerciseDebugLog("Exercise accepted successfully", ['user_id' => $user_id, 'session_id' => $session_id]);
                error_log("Exercise accepted for user $user_id, session $session_id");
                return true;
            } else {
                error_log("Exercise not found for user $user_id, session $session_id");
                exerciseDebugLog("Exercise not found in user data", [
                    'user_id' => $user_id, 
                    'session_id' => $session_id,
                    'available_sessions' => array_keys($exercises)
                ]);
                if (defined('ADMIN_ID')) {
                    sendMessage(ADMIN_ID, "❌ تمرین برای این کاربر و جلسه یافت نشد.\n\nکاربر: $user_id\nجلسه: $session_id\nموجود: " . implode(', ', array_keys($exercises)));
                }
                return false;
            }
        }
        
        // رد تمرین - مشابه تایید تمرین
        if (preg_match('/^exercise_reject_([0-9]+)_([0-9]+)$/', $data, $matches)) {
            $user_id = intval($matches[1]);
            $session_id = $matches[2];
            
            exerciseDebugLog("Exercise reject callback", ['user_id' => $user_id, 'session_id' => $session_id]);
            
            $user = getUserById($user_id);
            if (!$user) {
                error_log("User not found: $user_id");
                if (defined('ADMIN_ID')) {
                    sendMessage(ADMIN_ID, "❌ کاربر یافت نشد: $user_id");
                }
                return false;
            }
            
            $exercises = safeJsonDecode($user['exercises'] ?? null, []);
            
            // ✅ بررسی هر دو حالت: string و integer session_id
            $exercise_key = null;
            if (isset($exercises[$session_id])) {
                $exercise_key = $session_id;
            } elseif (isset($exercises[intval($session_id)])) {
                $exercise_key = intval($session_id);
            } elseif (isset($exercises[strval($session_id)])) {
                $exercise_key = strval($session_id);
            }
            
            if ($exercise_key !== null) {
                $exercises[$exercise_key]['status'] = 'rejected';
                $exercises[$exercise_key]['rejected_at'] = date('Y-m-d H:i:s');
                $user['exercises'] = json_encode($exercises, JSON_UNESCAPED_UNICODE);
                
                if (!saveUser($user)) {
                    error_log("Failed to save user after exercise reject");
                    if (defined('ADMIN_ID')) {
                        sendMessage(ADMIN_ID, "❌ خطا در ذخیره رد تمرین");
                    }
                    return false;
                }
                
                // پیدا کردن نام جلسه
                $session_title = $exercises[$exercise_key]['session_title'] ?? "";
                
                if (empty($session_title)) {
                    if (function_exists('loadSessions')) {
                        $sessions = loadSessions();
                        foreach ($sessions as $sess) {
                            if (intval($sess['id']) == intval($session_id)) {
                                $session_title = $sess['title'];
                                break;
                            }
                        }
                    }
                }
                
                if (empty($session_title)) {
                    $session_title = "جلسه شماره $session_id";
                }
                
                // اطلاع به کاربر
                sendMessage($user_id, "❌ تمرین شما برای جلسه <b>$session_title</b> رد شد.\n\nلطفاً آموزش را مجدد مطالعه کنید و تمرین را دوباره ارسال کنید.");
                
                // اطلاع به ادمین
                if (defined('ADMIN_ID')) {
                    sendMessage(ADMIN_ID, "❌ تمرین کاربر <code>$user_id</code> برای جلسه <b>$session_title</b> رد شد.");
                }
                
                exerciseDebugLog("Exercise rejected successfully", ['user_id' => $user_id, 'session_id' => $session_id]);
                error_log("Exercise rejected for user $user_id, session $session_id");
                return true;
            } else {
                error_log("Exercise not found for user $user_id, session $session_id");
                if (defined('ADMIN_ID')) {
                    sendMessage(ADMIN_ID, "❌ تمرین برای این کاربر و جلسه یافت نشد.");
                }
                return false;
            }
        }
        
        return false;
        
    } catch (Exception $e) {
        error_log("Error handling exercise callback: " . $e->getMessage());
        exerciseDebugLog("Exception in handleExerciseCallback", ['error' => $e->getMessage()]);
        if (defined('ADMIN_ID') && function_exists('sendMessage')) {
            sendMessage(ADMIN_ID, "❌ خطا در پردازش callback: " . $e->getMessage());
        }
        return false;
    }
}

/**
 * ✅ توابع سازگاری
 */
if (!function_exists('handleExerciseCallbackEnhanced')) {
    function handleExerciseCallbackEnhanced($data) {
        exerciseDebugLog("Enhanced callback received (redirecting to main handler)", ['data' => $data]);
        return handleExerciseCallback($data);
    }
}

if (!function_exists('handleExerciseViewCallback')) {
    function handleExerciseViewCallback($data) {
        exerciseDebugLog("View callback received (redirecting to main handler)", ['data' => $data]);
        return handleExerciseCallback($data);
    }
}

/**
 * بررسی اینکه آیا کاربر می‌تواند جلسه بعدی را ببیند
 */
function canSeeNextSession($user_id, $session_title) {
    try {
        if (!function_exists('loadSessions') || !function_exists('getUserById')) {
            return true; // در صورت نبود توابع، اجازه دسترسی بده
        }
        
        $sessions = loadSessions();
        $current_session = null;
        
        // پیدا کردن جلسه فعلی
        foreach ($sessions as $sess) {
            if ($sess['title'] == $session_title) {
                $current_session = $sess;
                break;
            }
        }
        
        if (!$current_session) {
            return true; // اگر جلسه پیدا نشد، اجازه دسترسی بده
        }
        
        $current_id = intval($current_session['id']);
        
        // اگر اولین جلسه است، همیشه قابل دسترسی است
        if ($current_id == 1) {
            return true;
        }
        
        // پیدا کردن جلسه قبلی
        $previous_session = null;
        foreach ($sessions as $sess) {
            if (intval($sess['id']) == ($current_id - 1)) {
                $previous_session = $sess;
                break;
            }
        }
        
        if (!$previous_session) {
            return true; // اگر جلسه قبلی پیدا نشد، اجازه دسترسی بده
        }
        
        // بررسی وضعیت تمرین جلسه قبلی
        $user = getUserById($user_id);
        $exercises = safeJsonDecode($user['exercises'] ?? null, []);
        
        $previous_session_id = intval($previous_session['id']);
        if (isset($exercises[$previous_session_id]) && 
            $exercises[$previous_session_id]['status'] == 'accepted') {
            return true;
        }
        
        return false;
        
    } catch (Exception $e) {
        error_log("Error checking session access for user $user_id: " . $e->getMessage());
        return true; // در صورت خطا، اجازه دسترسی بده
    }
}

/**
 * بررسی اینکه آیا جلسه‌ای آخرین جلسه است
 */
function isLastSession($session_id) {
    try {
        if (!function_exists('loadSessions')) {
            return false;
        }
        
        $sessions = loadSessions();
        if (empty($sessions)) {
            return false;
        }
        
        // مرتب‌سازی بر اساس ID
        usort($sessions, function($a, $b) {
            return intval($a['id']) - intval($b['id']);
        });
        
        $last_session = end($sessions);
        return intval($last_session['id']) == intval($session_id);
        
    } catch (Exception $e) {
        error_log("Error checking if session is last: " . $e->getMessage());
        return false;
    }
}

/**
 * دریافت آمار تمرین‌ها برای ادمین
 */
function getExerciseStats() {
    try {
        if (!function_exists('loadUsers') || !function_exists('loadSessions')) {
            return false;
        }
        
        $users = loadUsers();
        $sessions = loadSessions();
        
        $stats = [
            'total_users' => count($users),
            'total_exercises' => 0,
            'pending_exercises' => 0,
            'accepted_exercises' => 0,
            'rejected_exercises' => 0,
            'completed_users' => 0
        ];
        
        foreach ($users as $user) {
            $exercises = safeJsonDecode($user['exercises'] ?? null, []);
            $completed_count = 0;
            
            foreach ($exercises as $session_id => $exercise) {
                $stats['total_exercises']++;
                
                switch ($exercise['status'] ?? 'unknown') {
                    case 'pending':
                        $stats['pending_exercises']++;
                        break;
                    case 'accepted':
                        $stats['accepted_exercises']++;
                        $completed_count++;
                        break;
                    case 'rejected':
                        $stats['rejected_exercises']++;
                        break;
                }
            }
            
            // اگر تمام تمرین‌ها تایید شده باشند
            if ($completed_count == count($sessions)) {
                $stats['completed_users']++;
            }
        }
        
        return $stats;
        
    } catch (Exception $e) {
        error_log("Error getting exercise stats: " . $e->getMessage());
        return false;
    }
}

/**
 * ✅ دریافت لیست تمرین‌های در انتظار بررسی - نسخه کاملاً امن
 */
function getPendingExercises() {
    try {
        exerciseDebugLog("Getting pending exercises - START");
        
        if (!function_exists('loadUsers') || !function_exists('loadSessions')) {
            exerciseDebugLog("Required functions not found for getPendingExercises");
            return [];
        }
        
        $users = loadUsers();
        $sessions = loadSessions();
        $pending = [];
        
        exerciseDebugLog("Loaded data", ['users_count' => count($users), 'sessions_count' => count($sessions)]);
        
        foreach ($users as $user) {
            // ✅ پردازش ایمن JSON برای exercises
            $exercises = safeJsonDecode($user['exercises'] ?? null, []);
            
            if (!is_array($exercises) || empty($exercises)) {
                continue; // اگر تمرینی نداره، کاربر بعدی
            }
            
            exerciseDebugLog("Processing user exercises", [
                'user_id' => $user['id'],
                'user_name' => $user['first_name'] ?? 'نامشخص',
                'exercises_count' => count($exercises),
                'exercises_keys' => array_keys($exercises)
            ]);
            
            foreach ($exercises as $session_id => $exercise) {
                $exercise_status = $exercise['status'] ?? '';
                
                exerciseDebugLog("Checking exercise", [
                    'user_id' => $user['id'],
                    'session_id' => $session_id,
                    'status' => $exercise_status,
                    'has_answer' => isset($exercise['answer']),
                    'answer_length' => strlen($exercise['answer'] ?? ''),
                    'submitted_at' => $exercise['submitted_at'] ?? 'not_set'
                ]);
                
                if ($exercise_status == 'pending') {
                    // پیدا کردن نام جلسه
                    $session_title = $exercise['session_title'] ?? ''; // ✅ اول از داده ذخیره شده
                    
                    if (empty($session_title)) {
                        foreach ($sessions as $sess) {
                            if (intval($sess['id']) == intval($session_id)) {
                                $session_title = $sess['title'];
                                break;
                            }
                        }
                    }
                    
                    if (empty($session_title)) {
                        $session_title = "جلسه شماره $session_id";
                        exerciseDebugLog("Session title not found", ['session_id' => $session_id]);
                    }
                    
                    $pending_item = [
                        'user_id' => $user['id'],
                        'user_name' => $user['first_name'] ?? 'نامشخص',
                        'session_id' => intval($session_id),
                        'session_title' => $session_title,
                        'answer' => $exercise['answer'] ?? '',
                        'submitted_at' => $exercise['submitted_at'] ?? date('Y-m-d H:i:s')
                    ];
                    
                    $pending[] = $pending_item;
                    
                    exerciseDebugLog("Added pending exercise", $pending_item);
                }
            }
        }
        
        // مرتب‌سازی بر اساس زمان ارسال (جدیدترین اول)
        usort($pending, function($a, $b) {
            $time_a = strtotime($b['submitted_at']) ?: 0;
            $time_b = strtotime($a['submitted_at']) ?: 0;
            return $time_a - $time_b;
        });
        
        exerciseDebugLog("Pending exercises result", ['count' => count($pending)]);
        
        return $pending;
        
    } catch (Exception $e) {
        error_log("Error getting pending exercises: " . $e->getMessage());
        exerciseDebugLog("Exception in getPendingExercises", ['error' => $e->getMessage()]);
        return [];
    }
}
?>